package com.af.v4.system.runtime.controller.extend;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.excel.service.ExcelService;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.logic.service.LogicService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Map;

/**
 * CRUD动态表单配置控制器
 *
 * @author Mr.river
 */
@RestController
@RequestMapping("/crud")
public class CRUDFormConfigController {

    private static final Logger LOGGER = LoggerFactory.getLogger(CRUDFormConfigController.class);

    private final EntityService entityService;

    private final LogicService logicService;

    private final LiuLiConfigService liuLiConfigService;

    public CRUDFormConfigController(EntityService entityService, LogicService logicService, LiuLiConfigService liuLiConfigService) {
        this.entityService = entityService;
        this.logicService = logicService;
        this.liuLiConfigService = liuLiConfigService;
    }

    /**
     * 通过excel导入数据
     *
     * @param file            二进制文件
     * @param queryParamsName 查询配置名称
     * @return 执行结果
     */
    @Log(title = "通过excel导入数据", businessType = BusinessType.IMPORT)
    @PostMapping(value = "/importDataByExcel", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Object> importDataByExcel(@RequestParam("file") MultipartFile file,
                                       @RequestParam("queryParamsName") String queryParamsName) {
        try {
            JSONObject config = liuLiConfigService.get(queryParamsName, true);
            // 得到导入模板数据
            JSONArray excelImportTemplate = config.optJSONArray("excelImportTemplate");
            if (excelImportTemplate == null || excelImportTemplate.isEmpty()) {
                return R.fail("导入失败，没有配置导入列");
            }
            JSONObject apiSolt = config.optJSONObject("apiSlot");
            // 导入前插槽
            if (apiSolt != null && apiSolt.has("excelImportTemplateBefore")) {
                Object result = logicService.run(apiSolt.getString("excelImportTemplateBefore"), config);
                JSONObject resultObj = (JSONObject) result;
                if (resultObj.get("status").toString().equals("1")) {
                    return R.fail("导入失败，数据校验失败");
                }
            }
            JSONArray titles = new JSONArray();
            JSONArray columns = new JSONArray();
            // 得到所有的标题和字段
            excelImportTemplate.forEach(item -> {
                JSONObject itemObj = ((JSONObject) item);
                titles.putAll(itemObj.get("title"));
                columns.putAll(itemObj.get("column"));
            });
            // 导入excel得到数据，数据是以下标为key，数据值为value的map集合
            List<Map<Integer, String>> dataList = ExcelService.importExcel(file.getInputStream(), titles);
            excelImportTemplate.forEach(importTable -> {
                JSONObject importTableObj = ((JSONObject) importTable);
                // 得到表名
                String tableName = importTableObj.getString("tableName");
                LOGGER.info("导入表名：{}", tableName);
                // 批量保存数据
                dataList.forEach(data -> {
                    JSONObject entity = new JSONObject();
                    for (int i = 0; i < columns.length(); i++) {
                        entity.put(columns.getString(i), data.get(i));
                    }
                    LOGGER.info("导入数据：{}", entity);
                    try {
                        entityService.partialSave(tableName, entity);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                });
            });
            // 导入后插槽
            if (apiSolt != null && apiSolt.has("excelImportTemplateAfter")) {
                logicService.run(apiSolt.getString("excelImportTemplateAfter"),
                        new JSONObject().put("config", config).put("datas", dataList));
            }
            return R.ok("导入数据成功");
        } catch (Exception e) {
            LOGGER.error("导入数据失败", e);
            return R.fail("导入数据失败");
        }
    }
}
