package com.af.v4.system.runtime.service;

import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.expression.core.Program;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.resource.mapper.AbstractResourceMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 业务资源检测服务
 *
 * @author Mr.river
 */
@Service
public class ResCheckService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ResCheckService.class);

    private static final List<String> ERROR_ARRAY = new ArrayList<>();
    private static final String[] sentence = {
            "软件开发往往是这样：最开始的90%代码占用了开始的90%的开发时间；剩下10%代码同样需要90%的开发时间。",
            "这不是一个bug，这只是一个未列出来的特性。",
            "作为一个程序员，郁闷的事情是，⾯对一个代码块，却不敢去修改。更糟糕的是，这个代码块还是⾃⼰写的。",
            "在系统测试阶段查找并修复 Bug，花费的时间和工作量是让开发者自己找 Bug 的三倍。在正式上线后查找并修复 Bug，花费的时间和工作量是系统测试阶段的十倍。因此一定要让开发者自己做单元测试。",
            "不要站着调试。站着会让你的耐心减半，而且你需要集中所有精力。",
            "测试可以发现bug的存在，但不能证明bug不存在。",
            "新系统的每个新用户都会发现一类新的Bug。",
            "修复损坏的程序的第一步就是让它不断失败。",
            "有两种方法能写出没有错误的程序；但只有第三种好用。",
            "代码写的越急，程序跑得越慢。",
            "在所有的程序错误中，80%是语法错误，剩下20%里，80%是简单的逻辑错误，再剩下4%里，80%是指针错误，只有余下的0.8%才是困难的问题。"
    };
    private static final String[] prompt = {
            "生产环境下关闭Hibernate的自动建表，这是一个基本常识，好在我们已经这么做了。",
            "如何优雅的写Logic？使用Logic for IDEA提示插件，获得更智能的开发体验。",
            "不用怀念V3，往前走，别回头。",
            "将Logic文件像Java一样按照不同模块不同作用存放，这样的项目层级结构会更清晰。",
            "当你写出了超过150行的复杂SQL，尝试离开位置，到外面透透气，回来再想想，有没有更好的解决办法呢。",
            "如果你需要实现的业务涉及到使用新的框架、技术或者依赖，请与研发组确认，不要自己尝试增加任何依赖。",
            "如何成为一名好的程序员？偷懒吧，「懒」是程序员应有的美德，学着用软件工程思想来减少自己的重复低效工作。",
            "后端接口超过10秒就必须优化了，对于大量数据处理的业务，尝试用MQ，异步或者任务调度框架。",
            "代码可以重构，但重构前一定要备份，一定要备份，一定要备份。"
    };

    private final ApplicationUtils applicationUtils;

    private final List<AbstractResourceMapper<?>> abstractResourceMappers;

    public ResCheckService(ApplicationUtils applicationUtils, List<AbstractResourceMapper<?>> abstractResourceMappers) {
        this.applicationUtils = applicationUtils;
        this.abstractResourceMappers = abstractResourceMappers;
    }

    @PostConstruct
    public void checkAll() {
        EnvType envType = applicationUtils.getEnvType();
        if (envType != EnvType.DEV) {
            LOGGER.info("==================跳过资源检查==================");
            return;
        }
        LOGGER.info("==================开始资源强制检查==================");
        boolean throwError = false;
        StringBuilder msg = new StringBuilder("检查结果：");
        AbstractResourceMapper.RESOURCE_CHECK_FLAG.set(true);
        for (AbstractResourceMapper<?> mapper : abstractResourceMappers) {
            int errorNum = checkRes(mapper);
            if (errorNum > 0) {
                throwError = true;
                msg.append("\n * >>>ERROR: ").append(mapper.getResType())
                        .append("强制检查未通过，共有").append(errorNum).append("项错误");
            }
        }
        AbstractResourceMapper.RESOURCE_CHECK_FLAG.remove();
        if (throwError) {
            String errorWelcome = """
                    /***
                     * _ooOoo_
                     * o8888888o
                     * 88" . "88
                     * (| -_- |)
                     *  O\\ = /O
                     * ___/`---'\\____
                     * .   ' \\\\| |// `.
                     * / \\\\||| : |||// \\
                     * / _||||| -:- |||||- \\
                     * | | \\\\\\ - /// | |
                     * | \\_| ''\\---/'' | |
                     * \\ .-\\__ `-` ___/-. /
                     * ___`. .' /--.--\\ `. . __
                     * ."" '< `.___\\_<|>_/___.' >'"".
                     * | | : `- \\`.;`\\ _ /`;.`/ - ` : | |
                     * \\ \\ `-. \\_ __\\ /__ _/ .-` / /
                     * ======`-.____`-.___\\_____/___.-`____.-'======
                     * `=---='
                     * .............................................
                     * 佛曰：bug泛滥，我已瘫痪！
                    """;
            msg.append("\n * >>> 至理名言：").append(sentence[CommonTools.getRandomNumber(0, sentence.length - 1)]);
            errorWelcome = errorWelcome + "\n * " + msg + "\n";
            LOGGER.error(errorWelcome);
            for (int i = 0; i < ERROR_ARRAY.size(); i++) {
                StringBuilder item = new StringBuilder();
                LOGGER.error(item.append(" * ").append(i + 1).append(". ").append(ERROR_ARRAY.get(i)).toString());
            }
            LOGGER.error("==================资源强制检查完成==================\n");
            try {
                //此处睡眠是为了避免异步日志未输出完就抛出异常
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            throw new ServiceException("资源检查未通过，程序已强制中止");
        } else {
            String successWelcome = """
                    /***
                     *                    _ooOoo_
                     *                   o8888888o
                     *                   88" . "88
                     *                   (| -_- |)
                     *                    O\\ = /O
                     *                ____/`---'\\____
                     *              .   ' \\\\| |// `.
                     *               / \\\\||| : |||// \\
                     *             / _||||| -:- |||||- \\
                     *               | | \\\\\\ - /// | |
                     *             | \\_| ''\\---/'' | |
                     *              \\ .-\\__ `-` ___/-. /
                     *           ___`. .' /--.--\\ `. . __
                     *        ."" '< `.___\\_<|>_/___.' >'"".
                     *       | | : `- \\`.;`\\ _ /`;.`/ - ` : | |
                     *         \\ \\ `-. \\_ __\\ /__ _/ .-` / /
                     * ======`-.____`-.___\\_____/___.-`____.-'======
                     *                    `=---='
                     *
                     * .............................................
                     *          佛祖保佑             永无BUG
                     */""";
            LOGGER.info(successWelcome);
            LOGGER.info("小提示：" + prompt[CommonTools.getRandomNumber(0, prompt.length - 1)]);
        }
    }

    /**
     * 资源检测
     */
    private int checkRes(AbstractResourceMapper<? extends AbstractResourceMapper.CommonResource> mapper) {
        Map<String, ?> resMap = mapper.getAllMap();
        AtomicInteger i = new AtomicInteger(1);
        AtomicInteger success = new AtomicInteger(0);
        AtomicInteger error = new AtomicInteger(0);
        int size = resMap.size();
        LOGGER.info("检查" + mapper.getResType() + "资源，共 " + size + " 个...");
        resMap.forEach((key, value) -> {
            // 获取源程序内容
            AbstractResourceMapper.CommonResource resource = mapper.getResource(key);
            String source;
            String alias = resource.getAlias();
            String path = resource.getPath().toString();
            try {
                if (mapper.isSupportCompile()) {
                    source = resource.getSource();
                    Program pros = new Program(source);
                    // 解析
                    try {
                        pros.parse();
                        success.addAndGet(1);
                    } catch (RuntimeException e) {
                        String message = e.getMessage();
                        ERROR_ARRAY.add(mapper.getResType() + "资源[" + alias + "](" + path + ")编译错误，" + message);
                        error.addAndGet(1);
                    }
                } else {
                    success.addAndGet(1);
                }
            } catch (RuntimeException e) {
                ERROR_ARRAY.add(mapper.getResType() + "资源文件[" + alias + "]指向了一个不存在的位置：" + path);
                error.addAndGet(1);
            }
            i.addAndGet(1);
        });
        String resultMsg = mapper.getResType() + "资源检查完成，共 " + size + " 个,正确 " + success.get() + " 个,错误 " + error.get() + " 个";
        if (error.get() > 0) {
            LOGGER.error(resultMsg);
        } else {
            LOGGER.info(resultMsg);
        }
        return error.get();
    }
}
