package com.af.v4.system.runtime.controller;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.runtime.annotation.OpenApi;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.*;

/**
 * Logic控制器
 *
 * @author Mr.river
 */
@RestController
@RequestMapping("/logic")
public class LogicController {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicController.class);
    private static final String RESPONSE_HEADER = "header";
    private static final String RESPONSE_CONTENT = "content";
    private static final String RESPONSE_VALUE = "value";
    private final LogicService logicService;
    private final LogicMapper logicMapper;

    public LogicController(LogicService logicService, LogicMapper logicMapper) {
        this.logicService = logicService;
        this.logicMapper = logicMapper;
    }

    @Log(title = "调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Object run(@PathVariable("logicName") String logicName,
                      @RequestBody(required = false) String values,
                      HttpServletResponse response,
                      HttpServletRequest request) {
        String params = getRequestParams(values, request);
        return runMain(logicName, params, response);
    }

    @Log(title = "异步调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/async/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Object runAsync(@PathVariable("logicName") String logicName,
                           @RequestBody(required = false) String values,
                           HttpServletRequest request) {
        String params = getRequestParams(values, request);
        logicService.runAsync(logicName, params).whenComplete((o, e) -> {
            if (o != null) {
                LOGGER.info("异步调用结果：{}", o);
            }
        });
        return R.ok("ok");
    }

    @Log(title = "无token调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/openapi/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    @OpenApi
    public Object runByNoToken(@PathVariable("logicName") String logicName,
                               @RequestBody(required = false) String values,
                               HttpServletResponse response,
                               HttpServletRequest request) {
        return run(logicName, values, response, request);
    }

    @Log(title = "无token异步调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/openapi/async/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    @OpenApi
    public Object runAsyncByNoToken(@PathVariable("logicName") String logicName,
                                    @RequestBody(required = false) String values,
                                    HttpServletRequest request) {
        return runAsync(logicName, values, request);
    }

    /**
     * 请求参数格式化
     *
     * @param values  请求参数
     * @param request HttpServletRequest
     * @return 格式化后的请求参数
     */
    private String getValues(String values, HttpServletRequest request) {
        if (!values.startsWith("{")) {
            JSONObject paramJson = new JSONObject();
            paramJson.put("str", values);
            //加入请求头数据
            Enumeration<String> headerNames = request.getHeaderNames();
            if (headerNames != null) {
                JSONObject header = new JSONObject();
                while (headerNames.hasMoreElements()) {
                    String temp = headerNames.nextElement();
                    header.put(temp, request.getHeader(temp));
                }
                paramJson.put("header", header);
            }
            values = paramJson.toString();
        }
        return values;
    }

    /**
     * 获取请求参数
     *
     * @param values  请求体内容
     * @param request HttpServletRequest
     * @return 请求参数
     */
    private String getRequestParams(String values, HttpServletRequest request) {
        String params;
        if (request.getMethod().equals(HttpMethod.GET.name()) && Optional.ofNullable(values).isEmpty()) {
            Map<String, String[]> parameterMap = request.getParameterMap();
            JSONObject parameterMapObj = new JSONObject(parameterMap);
            params = parameterMapObj.toString();
        } else {
            params = getValues(values, request);
        }
        return params;
    }

    /**
     * Logic主执行入口
     *
     * @param logicName logic名称
     * @param values    参数
     * @param response  HttpServletResponse
     * @return 执行结果
     */
    private Object runMain(String logicName, String values, HttpServletResponse response) {
        Object runResult = logicService.run(logicName, values);
        if (runResult instanceof JSONObject object) {
            //如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
            if (object.has(RESPONSE_HEADER) && object.has(RESPONSE_CONTENT)) {
                JSONObject paramHeader = object.getJSONObject(RESPONSE_HEADER);
                for (String key : paramHeader.keySet()) {
                    response.setHeader(key, String.valueOf(paramHeader.get(key)));
                }
                runResult = object.get(RESPONSE_CONTENT);
            }
        }

        // 转换执行结果
        Object result;
        if (runResult instanceof JSONObject json) {
            result = json.toMap();
        } else if (runResult instanceof JSONArray array) {
            result = array.toList();
        } else if (runResult instanceof Map<?, ?> || runResult instanceof Collection<?>) {
            result = runResult;
        } else {
            if (runResult == null) {
                result = null;
            } else {
                Map<String, Object> objResult = new HashMap<>(1);
                objResult.put(RESPONSE_VALUE, runResult);
                result = objResult;
            }
        }

        // 如果为第三方回调接口，响应体不进行统一封装处理
        LogicMapper.LogicResource resource = logicMapper.getResource(logicName);
        if (resource.isCallbackApi()) {
            return result;
        }

        // 如果返回的是标准的响应格式，则根据响应码，返回对应的标准响应体
        if (result instanceof Map<?, ?> map) {
            if (map.containsKey("status")) {
                int status = (int) map.get("status");
                if (status == 200 && map.containsKey("data")) {
                    return R.ok(map.get("data"));
                } else if (status != 200 && map.containsKey("msg")) {
                    return R.fail(status, (String) map.get("msg"));
                }
            }
        }

        return R.ok(result);
    }
}
