package com.af.v4.system.restful.sql;

import com.af.v4.system.restful.transaction.SessionPool;
import org.hibernate.Session;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.query.NativeQuery;
import org.hibernate.transform.ResultTransformer;
import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * SQL执行类
 *
 * @author Mr.river
 */
@Component
public class SqlAction implements TransformerSupport {

    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    private final SessionPool sessionPool;

    private final ThreadLocal<Boolean> useStandardTransformerThreadLocal = new ThreadLocal<>();

    public SqlAction(SessionPool sessionPool) {
        this.sessionPool = sessionPool;
    }

    public JSONArray query(String name, String sql) {
        return query(name, sql, 0, 1000);
    }

    public JSONArray queryAll(String name, String sql) {
        return query(name, sql, null, null);
    }

    public JSONArray query(String name, String sql, Integer pageNo,
                           Integer pageSize) {
        return (JSONArray) sqlMonitor(() -> {
            ResultTransformer resultTransformer;
            Boolean userStandardTrans = useStandardTransformerThreadLocal.get();
            boolean useStandardTransformer = userStandardTrans != null && userStandardTrans;
            if(useStandardTransformer){
                useStandardTransformerThreadLocal.remove();
                resultTransformer = StandardAliasTransformer.INSTANCE;
            } else {
                resultTransformer = AliasTransformer.INSTANCE;
            }
            HibernateSQLCall sqlCall = new HibernateSQLCall(sql, pageNo, pageSize, resultTransformer);
            return sqlCall.doInHibernate(sessionPool.getSession());
        }, name, sql);
    }

    public int bulkSQLUpdate(String name, String sql) {
        return (int) sqlMonitor(() -> {
            Session session = sessionPool.getSession();
            NativeQuery<?> queryObject = session.createNativeQuery(sql);
            return queryObject.executeUpdate();
        }, name, sql);
    }

    private Object sqlMonitor(Process process, String name, String sql) {
        long begin = System.currentTimeMillis();
        try {
            return process.apply();
        } finally {
            long end = System.currentTimeMillis();
            LOGGER.info("维护信息:执行SQL["+name+"]耗时：" + (end - begin) + "ms:{\n" + sql + "\n}");
        }
    }

    @Override
    public void useStandardTransformer() {
        this.useStandardTransformerThreadLocal.set(true);
    }

    @FunctionalInterface
    interface Process {
        Object apply();
    }

    private static class HibernateSQLCall implements HibernateCallback<JSONArray> {
        final String sql;
        final Integer page;
        final Integer rows;
        public ResultTransformer transformer;

        public HibernateSQLCall(String sql, Integer page, Integer rows, ResultTransformer transformer) {
            this.sql = sql;
            this.page = page;
            this.rows = rows;
            this.transformer = transformer;
        }

        @Override
        public JSONArray doInHibernate(Session session) {
            NativeQuery<?> q = session.createNativeQuery(sql);
            q.setResultTransformer(transformer);
            try {
                List<?> list;
                if(page == null || rows == null){
                    list = q.list();
                } else {
                    list = q.setFirstResult(page * rows).setMaxResults(rows).list();
                }
                if (transformer instanceof StandardAliasTransformer){
                    return new JSONArray().putAll(list);
                } else {
                    return new JSONArray(list);
                }
            } catch (SQLGrammarException ex) {
                // 把sql语句添加到异常信息中
                String msg = "sql:\n" + sql + "\n" + ex.getMessage();
                throw new SQLGrammarException(msg, ex.getSQLException());
            }
        }
    }
}
