package com.af.v4.system.restful.service;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.expression.Expression;
import com.af.v4.system.restful.exception.FileNotFoundException;
import com.af.v4.system.restful.mapper.AbstractResourceMapper;
import com.af.v4.system.restful.mapper.ResourceMapper;
import com.af.v4.system.restful.mapper.SqlMapper;
import com.af.v4.system.restful.sql.SqlAction;
import com.af.v4.system.restful.sql.TransformerSupport;
import com.af.v4.system.restful.utils.enums.ResourceType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Iterator;

@Service
public class SqlService implements TransformerSupport {
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlService.class);

    private final ResourceMapper resourceMapper;

    private final SqlMapper sqlMapper;

    private final SqlAction sqlAction;

    public SqlService(ResourceMapper resourceMapper, SqlMapper sqlMapper, SqlAction sqlAction) {
        this.resourceMapper = resourceMapper;
        this.sqlMapper = sqlMapper;
        this.sqlAction = sqlAction;
    }

    /**
     * 获取SQL文件的编译后SQL
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return SQL语句
     */
    public String getSQL(String name, JSONObject params) {
        return this.call(name, params);
    }


    /**
     * 获取SQL文件的合计执行结果
     *
     * @param name SQL映射名
     * @param params 参数
     * @return 结果集
     */
    public JSONObject queryTotal(String name, JSONObject params){
        // 获取参数，求和字段等内容
        Object sums = null;
        if (params.has("sums")) {
            sums = params.get("sums");
        }
        // 产生sql语句编译后的结果
        String sql = this.call(name, params);

        // 求和时，order by会导致sql错误，过滤掉order by部分。
        sql = filterOutOrderBy(sql, sums);
        JSONArray array = sqlAction.query(name, sql);
        return array.getJSONObject(0);
    }

    /**
     * 获取SQL语句的合计执行结果
     *
     * @param name SQL映射名
     * @param str json字符串，取里面的data作为参数
     * @return 结果集
     */
    public JSONObject queryTotal(String name, String str) {
        // 获取参数，求和字段等内容
        JSONObject param;
        Object sums = null;
        if (str != null && !str.isEmpty()) {
            JSONObject json = new JSONObject(str);
            if (json.has("data")) {
                param = json.getJSONObject("data");
            } else {
                param = new JSONObject();
            }
            if (json.has("sums")) {
                sums = json.get("sums");
            }
        } else {
            param = new JSONObject();
        }
        // 产生sql语句编译后的结果
        String sql = this.call(name, param);

        // 求和时，order by会导致sql错误，过滤掉order by部分。
        sql = filterOutOrderBy(sql, sums);
        JSONArray array = sqlAction.query(name, sql);
        return array.getJSONObject(0);
    }

    /**
     * 执行SQL文件查询
     * @param name SQL映射名
     * @param params 参数
     * @param pageNo 页码
     * @param pageSize 每页大小
     * @return 结果集
     */
    public JSONArray query(String name, JSONObject params, Integer pageNo, Integer pageSize) {
        // 产生SQL语句编译后的结果
        String sql = this.call(name, params);

        JSONArray array = sqlAction.query(name, sql, pageNo - 1, pageSize);
        writeSqlResult(name, array);
        return array;
    }
    public JSONArray query(String name, JSONObject params, Integer pageSize) {
        return pageSize <= 0 ? query(name, params, 1, 9999999) : query(name, params, 1, pageSize);
    }
    public JSONArray query(String name, JSONObject params) {
        return query(name, params, 1, 1000);
    }

    /**
     * 执行SQL文件查询（参数为json字符串，取里面的data作为参数）
     * @param name SQL映射名
     * @param str json字符串，取里面的data作为参数
     * @param pageNo 页码
     * @param pageSize 每页大小
     * @return 结果集
     */
    public JSONArray query(String name, String str, Integer pageNo, Integer pageSize) {
        // pageNo小于0， 纠正成1
        if (pageNo <= 0) {
            pageNo = 1;
        }

        // pageSize小于0，纠正成1, 解除1000限制
        if (pageSize < 1) {
            pageSize = 9999999;
        }

        // 拿到json对象参数
        JSONObject param;
        if (str != null && !str.isEmpty()) {
            param = new JSONObject(str);
            param = param.getJSONObject("data");
        } else {
            param = new JSONObject();
        }
        return query(name, param, pageNo, pageSize);
    }
    public JSONArray query(String name, String str, Integer pageSize) {
        return pageSize <= 0 ? query(name, str, 1, 9999999) : query(name, str, 1, pageSize);
    }
    public JSONArray query(String name, String str) {
        return query(name, str, 1, 1000);
    }

    /**
     * 具名SQL语句查询
     * @param name SQL查询标识名
     * @param sql SQL语句
     * @return 结果集
     *
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    public JSONArray querySQL(String name, String sql) {
        return querySQL(name, sql, 1, 1000);
    }

    public JSONArray querySQL(String name, String sql, Integer pageSize) {
        return pageSize <= 0 ? querySQL(name, sql, 1, 9999999) : querySQL(name, sql, 1, pageSize);
    }

    public JSONArray querySQL(String name, String sql, Integer pageNo, Integer pageSize) {
        JSONArray array = sqlAction.query(name, sql, pageNo - 1, pageSize);
        writeSqlResult(name, array);
        return array;
    }

    /**
     * 执行SQL文件增删改
     * @param name SQL映射名
     * @param params 参数
     * @return 执行结果
     */
    public Integer update(String name, JSONObject params) {
        // 产生sql语句编译后的结果
        String sql = this.call(name, params);
        return updateSQL(name, sql);
    }

    /**
     * 具名SQL语句增删改
     * @param name SQL查询标识名
     * @param sql SQL语句
     * @return 影响行数
     *
     * @apiNote 注意：不推荐依赖该方法作为SQL查询，因为该方法没有校验name参数的唯一性，这会给排查SQL问题的同事带来麻烦
     */
    public Integer updateSQL(String name, String sql) {
        return sqlAction.bulkSQLUpdate(name, sql);
    }


    /**
     * 基于JSON对象的call函数
     */
    public String call(String sqlName, JSONObject json) {
        // 获取原始sql语句
        AbstractResourceMapper.CommonResource<String> commonResource = sqlMapper.getValue(sqlName);
        sqlName = commonResource.getAlias();
        String path = commonResource.getPath();
        if (path == null) {
            throw new ServiceException("sql语句未注册！" + sqlName);
        }
        String sql;
        try {
            sql = resourceMapper.getString(ResourceType.SQL, sqlName, path);
            //如果使用了新格式化参数写法，需要替换为解析语法树支持的写法
            if (sql.contains("${")) {
                sql = sql.replace("${", "{");
            }
        } catch (FileNotFoundException e) {
            throw new ServiceException(path + ".文件无配置");
        }
        // 把自身注册到执行环境中
        json.put("sql", this);
        sql = Expression.run(sql, json.toMap()).toString();
        return sql;
    }

    /**
     * 过滤order by子句，产生求和结果
     */
    private String filterOutOrderBy(String source, Object sums) {
        int idx = source.toLowerCase().lastIndexOf("order by");
        if (idx == -1) {
            idx = source.length() - 1;
        }
        StringBuilder sql = new StringBuilder("select ");
        // 如果有求和部分，产生求和部分的语句
        if (sums != null) {
            if (sums instanceof JSONArray arraySums) {
                for (int i = 0; i < arraySums.length(); i++) {
                    String name = (String) arraySums.get(i);
                    sql.append("sum(").append(name).append(") ").append(name).append(", ");
                }
            } else if (sums instanceof JSONObject objSums) {
                Iterator<String> keys = objSums.keys();
                while (keys.hasNext()) {
                    String name = keys.next();
                    String value = objSums.getString(name);
                    sql.append("sum(").append(value).append(") ").append(name).append(", ");
                }
            }
        }
        if (idx != -1) {
            sql.append("count(*) n, 1 placeholder from ( ").append(source, 0, idx).append(") t_");
        }
        return sql.toString();
    }

    /**
     * 打印SQL执行结果
     * @param name SQL名称
     * @param array 执行结果
     */
    private void writeSqlResult(String name, JSONArray array){
        String result = array.toString();
        int length = result.length();
        if(length > 2000){
            result = result.substring(0, 2000) + "...后续还有" + (length - 2000) + "个字符";
        }
        LOGGER.debug("SQL[" + name + "] Result: \n" + result);
    }

    @Override
    public void useStandardTransformer() {
        this.sqlAction.useStandardTransformer();
    }
}
