package com.af.v4.system.restful.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * 异常日志记录管理
 */
@Component
public class ExceptionLogService {

    private static final String HIBERNATE_SQL_EXCEPTION = "org.hibernate.exception.SQLGrammarException";

    private static final String EXPRESSION_EXCEPTION = "com.af.v4.system.expression.exception.ExpressionException";
    private static final Logger LOGGER = LoggerFactory.getLogger(ExceptionLogService.class);

    /**
     * 获取异常信息
     *
     * @param errorInfo 异常原始数据
     * @param type      业务类型
     * @return 异常信息
     */
    public static String getErrorInfo(String errorInfo, BusinessType type) {
        String result;
        try {
            String defaultHeader = "ERROR!======\n";
            String defaultHeaderEnd = "\nERROR_END!=====";
            StringBuilder builder = new StringBuilder(defaultHeader);
            if (type == BusinessType.LOGIC) {
                //Logic报错堆栈截取思路如下：
                //将堆栈分成两部分
                //主要部分：Logic异常报错时，会出现com.af.v4.system.expression.exception.ExpressionException异常，通过该关键词递归截取
                //结尾部分：截取Caused by，获取根堆栈信息
                appendLogicErrorInfo(builder, errorInfo);
            } else {
                //Sql报错堆栈截取思路如下：
                //将堆栈分成两部分
                //主要部分：SQL异常报错时，会出现org.hibernate.exception.SQLGrammarException异常，通过该关键词递归截取
                //结尾部分：截取Caused by，获取根堆栈信息
                appendSqlErrorInfo(builder, errorInfo);
            }
            builder.append(defaultHeaderEnd);
            result = builder.toString();
            if (result.equals(defaultHeader + defaultHeaderEnd)) {
                result = errorInfo;
            }
        } catch (Exception e) {
            LOGGER.error("***报错堆栈截取报错***", e);
            result = errorInfo;
        }
        return result;
    }

    private static void appendSqlErrorInfo(StringBuilder result, String errorInfo) {
        String keySentences = HIBERNATE_SQL_EXCEPTION + ": sql:\n";
        //堆栈内容
        int ignoreIndex = errorInfo.indexOf(keySentences);
        if (ignoreIndex != -1) {
            int endIndex = errorInfo.indexOf("could not extract ResultSet");
            if (endIndex != -1) {
                String head = errorInfo.substring(ignoreIndex + keySentences.length(), endIndex);
                result.append(head).append("=====\n");
            }
        }
        //截取结尾部分
        ignoreIndex = errorInfo.lastIndexOf("Caused by:");
        if (ignoreIndex != -1) {
            result.append(errorInfo.substring(ignoreIndex));
        }
    }

    private static void appendLogicErrorInfo(StringBuilder result, String errorInfo) {
        String keySentences = EXPRESSION_EXCEPTION + ": ";
        //堆栈内容
        int ignoreIndex = errorInfo.indexOf(keySentences);
        if (ignoreIndex != -1) {
            errorInfo = errorInfo.substring(ignoreIndex + keySentences.length());
            int ignoreIndexEnd = errorInfo.indexOf("at ");
            if (ignoreIndexEnd != -1) {
                result.append(errorInfo, 0, ignoreIndexEnd).append("\n=====\n");
                errorInfo = errorInfo.substring(ignoreIndexEnd + 3);
                appendLogicErrorInfo(result, errorInfo);
            }
        } else {
            //截取结尾部分
            ignoreIndex = errorInfo.lastIndexOf("Caused by:");
            if (ignoreIndex != -1) {
                result.append(errorInfo.substring(ignoreIndex));
            }
        }
    }

    enum BusinessType {
        LOGIC,
        SQL
    }
}
