package com.af.v4.system.restful.mapper;

import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.plugins.json.JsonTools;
import com.af.v4.system.restful.service.QueryParamsService;
import com.af.v4.system.restful.sql.SqlAction;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.File;
import java.util.Objects;

/**
 * 查询参数映射器
 *
 * @author Mr.river
 */
@Component
public class QueryParamsMapper {
    public static final String QUERY_PARAMS_IS_LOADED_KEY = "QUERY_PARAMS_IS_LOADED";
    public static final String NO_LOAD_FLAG = "0";
    public static final String LOADED_FLOG = "1";
    public static final String QUERY_PARAMS_PREFIX = "QP_";
    private static final Logger LOGGER = LoggerFactory.getLogger(QueryParamsMapper.class);
    private static final String QUERY_SQL = "SELECT * FROM T_QUERY_PARAMS";
    private static final String PUBLIC_QUERY_PARAMS_DIRECTORY_PATH = "/system/queryparams/";

    private final RedisService redisService;
    private final SqlAction sqlAction;

    private final QueryParamsService queryParamsService;

    public QueryParamsMapper(RedisService redisService, SqlAction sqlAction, QueryParamsService queryParamsService) {
        this.redisService = redisService;
        this.sqlAction = sqlAction;
        this.queryParamsService = queryParamsService;
    }

    @PostConstruct
    public void load() {
        redisService.lock(QUERY_PARAMS_IS_LOADED_KEY, () -> {
            Object isLoaded = redisService.get(QUERY_PARAMS_IS_LOADED_KEY);
            if (isLoaded == null || NO_LOAD_FLAG.equals(isLoaded)) {
                JSONArray result = null;
                try {
                    result = sqlAction.query("QueryParamsMapper@load", QUERY_SQL);
                } catch (Exception e) {
                    LOGGER.warn("查询配置表(t_query_params)查询失败，不再加载查询配置");
                }
                if (result != null) {
                    for (Object o : result) {
                        JSONObject item = (JSONObject) o;
                        if (item.has("f_content")) {
                            JSONObject compileSource = queryParamsService.getQueryParamsInfo(item.getString("f_content"));
                            redisService.set(QUERY_PARAMS_PREFIX + item.getString("f_name"), compileSource);
                        }
                    }
                }
                //加载通用公共配置
                File directory = new File(Objects.requireNonNull(QueryParamsMapper.class.getResource(PUBLIC_QUERY_PARAMS_DIRECTORY_PATH)).getPath());
                File[] fileList = directory.listFiles();
                if(fileList != null && fileList.length > 0){
                    for(File file : fileList){
                        String fileName = file.getName();
                        JSONObject configItem = JsonTools.readJsonFile(PUBLIC_QUERY_PARAMS_DIRECTORY_PATH + "/" + fileName);
                        JSONObject compileSource = queryParamsService.getQueryParamsInfo(configItem.toString());
                        redisService.set(QUERY_PARAMS_PREFIX + fileName.substring(0, fileName.indexOf(".")), compileSource);
                    }
                }
                //修改加载标识
                redisService.set(QUERY_PARAMS_IS_LOADED_KEY, LOADED_FLOG);
                LOGGER.debug("查询配置预加载完成");
            } else {
                LOGGER.debug("跳过加载查询配置表");
            }
        });
    }

    /**
     * 查询数据源缓存清理
     */
    public void clearQueryParamsCache() {
        redisService.lock(QueryParamsMapper.QUERY_PARAMS_IS_LOADED_KEY, () -> redisService.set(QueryParamsMapper.QUERY_PARAMS_IS_LOADED_KEY, QueryParamsMapper.NO_LOAD_FLAG));
        this.load();
    }
}
