package com.af.v4.system.restful.mapper;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.plugins.io.IOTools;
import com.af.v4.system.restful.config.SystemConfig;
import com.af.v4.system.restful.utils.SpringBeansPour;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RestController;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

/**
 * 抽象资源映射器
 */
@Component
public abstract class AbstractResourceMapper<T> implements ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractResourceMapper.class);
    private static ApplicationContext applicationContext;
    private final ModuleMapper moduleMapper;
    private final SystemConfig systemConfig;
    /**
     * 资源列表
     */
    protected Map<String, Map<String, T>> map;

    protected AbstractResourceMapper(ModuleMapper moduleMapper, SystemConfig systemConfig) {
        this.moduleMapper = moduleMapper;
        this.systemConfig = systemConfig;
    }

    /**
     * 资源结果集
     * @param <T> 资源数据类型
     */
    public static class CommonResource<T> {
        private final String alias;
        private final T path;


        CommonResource(String alias, T path) {
            this.alias = alias;
            this.path = path;
        }


        public String getAlias() {
            return alias;
        }

        public T getPath() {
            return path;
        }
    }

    /**
     * 获取资源类型
     *
     * @return 资源类型
     */
    protected abstract String getResType();

    /**
     * 获取资源文件名
     *
     * @return 资源文件名
     */
    protected abstract String getFileName();

    /**
     * 获取资源文件夹名
     *
     * @return 资源文件夹名
     */
    protected abstract String getFolderName();

    /**
     * 加载资源列表
     */
    protected void loadMap() {
        if (map == null) {
            map = new HashMap<>(36);
            putMapByRoot();
            moduleMapper.getMap().forEach((key, value) -> {
                String name = value.get("name");
                String path = value.get("path");
                try {
                    putMapByModule(name, path);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
    }

    /**
     * 加载根目录资源到列表
     */
    protected void putMapByRoot() {
        putMapByModule(null, null);
    }

    /**
     * 加载模块资源到列表
     *
     * @param name   模块名
     * @param parent 父模块名
     */
    protected void putMapByModule(String name, String parent) {
        String str;
        String fileName = getFileName();
        String resType = getResType();
        if (parent == null) {
            if (name == null) {
                str = fileName;
            } else {
                str = name + "/" + fileName;
            }
        } else {
            str = parent + "/" + name + "/" + fileName;
        }
        IOTools.getStream(str, stream -> {
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(stream);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
            Element root = document.getRootElement();
            for (Iterator<Element> item = root.elementIterator(resType); item.hasNext(); ) {
                Element element = item.next();
                appendRes(element, name, parent);
            }
        });
    }

    /**
     * 获取资源Map
     *
     * @return 全部资源
     */
    public Map<String, Map<String, T>> getAllMap() {
        return map;
    }

    /**
     * 根据Key获取值
     *
     * @param key KEY
     * @return value
     */
    public CommonResource<T> getValue(String key) {
        String alias = getAlias(key);
        if(map.containsKey(alias)){
            return new CommonResource<>(alias, map.get(alias).get("value"));
        } else {
            return new CommonResource<>(alias, null);
        }
    }

    /**
     * 根据key获取资源别名
     * @param key key
     * @return 资源别名
     */
    private String getAlias(String key){
        String versionAliasValue = systemConfig.getVersionAliasValue();
        if(versionAliasValue.equals(SystemConfig.STANDARD_VERSION_VALUE)){
            return key;
        } else {
            String alias = key + "@" + versionAliasValue;
            if (map.containsKey(alias)) {
                return alias;
            } else {
                return key;
            }
        }
    }

    /**
     * 根据Key获取对应资源Map
     * @param key KEY
     * @return 资源Map
     */
    public Map<String, T> getRes(String key){
        String alias = getAlias(key);
        return map.getOrDefault(alias, null);
    }

    /**
     * 追加资源
     *
     * @param element          XML元素
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    protected Map<String, T> appendRes(Element element, String moduleName, String parentModuleName) {
        String alias = element.attribute("alias").getValue();
        // 如果是分公司资源，修改注册的别名
        if(Objects.equals(parentModuleName, "filiale")){
            alias = alias + "@" + moduleName;
        }
        // 验证别名是否重复
        if (map.containsKey(alias)) {
            String resType = getResType();
            throw new ServiceException(resType + "别名[" + alias + "]已存在");
        }
        // 设置属性
        Map<String, T> res = new HashMap<>(4);
        T value;
        Attribute pathAttribute = element.attribute("path");
        if (pathAttribute != null) {
            String path = pathAttribute.getValue();
            String folderName = getFolderName();
            path = (parentModuleName == null ?
                    moduleName + "/" + folderName + "/" :
                    parentModuleName + "/" + moduleName + "/" + folderName + "/") + path;
            value = (T) path;
        } else {
            String className = element.attribute("class").getValue();
            Object obj = getClassByName(className, moduleName);
            value = (T) obj;
        }
        if (value != null) {
            res.put("alias", (T) alias);
            res.put("value", value);
            map.put(alias, res);
        }
        return res;
    }

    private Object getClassByName(String className, String moduleName) {
        try {
            Class<?> c = Class.forName(className);
            Object obj;
            // 如果是spring的bean，让spring取
            if (isSpringBean(c)) {
                obj = applicationContext.getBean(c);
            } else {
                obj = c.getDeclaredConstructor().newInstance();
                // 如果实现了插件Bean注入工具
                if (obj instanceof SpringBeansPour) {
                    ((SpringBeansPour) obj).initBeans(applicationContext);
                }
            }
            return obj;
        } catch (ClassNotFoundException | NoClassDefFoundError ex) {
            LOGGER.warn("未找到类, 模块=" + moduleName + ", 类=" + ex.getMessage());
            return null;
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    private boolean isSpringBean(Class<?> c) {
        for (Annotation annotation : c.getAnnotations()) {
            if (annotation.annotationType() == Component.class ||
                    annotation.annotationType() == Service.class ||
                    annotation.annotationType() == RestController.class
            ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Spring启动的时候会自动调用下面的方法设置ApplicationContext的值
     */
    @Override
    public void setApplicationContext(ApplicationContext ctx) throws BeansException {
        applicationContext = ctx;
        //初始化即加载,之所以放到该方法里，是因为static静态块比该方法先执行，会导致applicationContext对象为null
        loadMap();
    }
}
