package com.af.v4.system.restful.controller;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.restful.mapper.ResourceMapper;
import com.af.v4.system.restful.service.LogicService;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * Logic控制器
 *
 * @author Mr.river
 */
@RestController
@RequestMapping("/logic")
public class LogicController {
    private final LogicService logicService;

    public LogicController(LogicService logicService) {
        this.logicService = logicService;
    }

    @Log(title = "调用Logic(GET)", businessType = BusinessType.LOGIC)
    @GetMapping(value = "{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Object> runGet(@PathVariable("logicName") String logicName,
                            HttpServletResponse response,
                            HttpServletRequest request) {
        Map<String, String[]> parameterMap = request.getParameterMap();
        JSONObject params = new JSONObject(parameterMap);
        String values = params.toString();
        return runMain(logicName, values, response);
    }

    @Log(title = "调用Logic", businessType = BusinessType.LOGIC)
    @PostMapping(value = "{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Object> run(@PathVariable("logicName") String logicName,
                         @RequestBody String values,
                         HttpServletResponse response,
                         HttpServletRequest request) {
        values = getValues(values, request);
        return runMain(logicName, values, response);
    }

    /**
     * 清理所有缓存
     *
     * @return ok
     */
    @Log(title = "清理所有缓存", businessType = BusinessType.CLEAR_CACHE)
    @GetMapping(value = "/clearall", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<String> clearAllCache() {
        ResourceMapper.clearLogicCache();
        ResourceMapper.clearSqlCache();
        return R.ok("ok");
    }

    /**
     * 请求参数格式化
     * @param values 请求参数
     * @param request HttpServletRequest
     * @return 格式化后的请求参数
     */
    private String getValues(String values, HttpServletRequest request) {
        if (!values.startsWith("{")) {
            JSONObject paramJson = new JSONObject();
            paramJson.put("str", values);
            //加入请求头数据
            Enumeration<String> headerNames = request.getHeaderNames();
            if (headerNames != null) {
                JSONObject header = new JSONObject();
                while (headerNames.hasMoreElements()) {
                    String temp = headerNames.nextElement();
                    header.put(temp, request.getHeader(temp));
                }
                paramJson.put("header", header);
            }
            values = paramJson.toString();
        }
        return values;
    }

    /**
     * Logic主执行入口
     * @param logicName logic名称
     * @param values 参数
     * @param response HttpServletResponse
     * @return 执行结果
     */
    private R<Object> runMain(String logicName, String values, HttpServletResponse response) {
        Object runResult = logicService.runMain(logicName, values);

        if (runResult instanceof JSONObject object) {
            //如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
            if (object.has("header") && object.has("content")) {
                JSONObject paramHeader = object.getJSONObject("header");
                for (String key : paramHeader.keySet()) {
                    response.setHeader(key, String.valueOf(paramHeader.get(key)));
                }
                runResult = object.get("content");
            }
        }

        // 转换执行结果
        Object result;
        if (runResult instanceof JSONObject json) {
            result = json.toMap();
        } else if (runResult instanceof JSONArray array) {
            result = array.toList();
        } else if (runResult instanceof Map<?, ?> || runResult instanceof Collection<?>) {
            result = runResult;
        } else {
            Map<String, Object> objResult = new HashMap<>(1);
            objResult.put("value", runResult.toString());
            result = objResult;
        }

        return R.ok(result);
    }
}
