package com.af.v4.system.restful.service;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.plugins.core.CommonTools;
import com.af.v4.system.plugins.io.IOTools;
import com.af.v4.system.restful.config.SystemConfig;
import com.af.v4.system.restful.enums.entity.ColumnTypeEnum;
import com.af.v4.system.restful.enums.entity.DialectTypeEnum;
import com.af.v4.system.restful.enums.entity.IDTypeEnum;
import com.af.v4.system.restful.sql.SqlAction;
import com.af.v4.system.restful.sql.dynamic.DynamicDataSource;
import com.af.v4.system.restful.transaction.SessionPool;
import com.af.v4.system.restful.utils.Pair;
import com.alibaba.druid.pool.DruidDataSource;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.id.*;
import org.hibernate.id.enhanced.SequenceStyleGenerator;
import org.hibernate.internal.SessionImpl;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.metamodel.spi.MetamodelImplementor;
import org.hibernate.persister.entity.AbstractEntityPersister;
import org.hibernate.persister.entity.EntityPersister;
import org.hibernate.query.NativeQuery;
import org.hibernate.type.CollectionType;
import org.hibernate.type.ManyToOneType;
import org.hibernate.type.OneToOneType;
import org.hibernate.type.Type;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.io.*;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.*;

@Service
@Transactional(rollbackFor = Exception.class)
public class EntityService {
    private static final Logger LOGGER = LoggerFactory.getLogger(EntityService.class);
    private static final Map<String, Map<String, String>> SUBCLASS_MODULE_MAP;
    private static final HashMap<String, Map<String, Object>> META_MAP = new HashMap<>();
    private static final HashMap<String, String> ENTITY_LIFT_MAP = new HashMap<>();

    static {
        SUBCLASS_MODULE_MAP = new LinkedHashMap<>();
        JSONArray json = SystemConfig.Hibernate.getSubClassArray();
        if (json != null) {
            for (Object obj : json) {
                String name = String.valueOf(obj);
                Map<String, String> module = new HashMap<>(json.length());
                module.put("name", name);
                SUBCLASS_MODULE_MAP.put(name, module);
            }
        }
    }

    private final SessionPool sessionPool;
    private final SqlAction sqlAction;


    public EntityService(SessionPool sessionPool, SqlAction sqlAction) {
        this.sessionPool = sessionPool;
        this.sqlAction = sqlAction;
    }

    /**
     * 获取方言类型
     */
    public static DialectTypeEnum getDialectType() {
        DruidDataSource dataSource = DynamicDataSource.getDruidDataSource();
        DialectTypeEnum dialectType = DialectTypeEnum.toType(dataSource.getDriverClassName());
        if (dialectType == DialectTypeEnum.DIALECT_NOT_SUPPORTED) {
            throw new RuntimeException("数据库方言配置异常。");
        }
        return dialectType;
    }

    /**
     * 获取方言类型字符串
     */
    public static String getDialectTypeStr() {
        return getDialectType().getValue();
    }

    /**
     * 获取继承关系模块配置文件
     */
    public static Map<String, Map<String, String>> getSubClassMap() {
        return SUBCLASS_MODULE_MAP;
    }

    @PostConstruct
    public void init() {
        // 获取元数据
        loadMetaData();
    }

    /**
     * 删除单个实体
     *
     * @param entityName 实体名
     * @param id         主键值
     * @return 影响行数
     */
    public Integer delete(String entityName, Object id) {
        Map<String, Object> map = META_MAP.get(entityName);
        String sql = "DELETE FROM " + map.get("tableName") + " WHERE " + map.get("idName") + " = '" + id + "'";
        return sqlAction.bulkSQLUpdate("EntityService@Delete", sql);
    }

    /**
     * 批量删除实体
     *
     * @param entityName 实体名
     * @param ids        主键值集合
     * @return 影响行数
     */
    public Integer deleteAllByIds(String entityName, JSONArray ids) {
        Map<String, Object> map = META_MAP.get(entityName);
        String idStr = CommonTools.union(ids);
        String sql = "DELETE FROM " + map.get("tableName") + " WHERE " + map.get("idName") + " IN (" + idStr + " )";
        return sqlAction.bulkSQLUpdate("EntityService@deleteAllByIds",sql);
    }

    /**
     * 根据ID集合查询所有数据
     *
     * @param columns    列字段
     * @param entityName 实体名
     * @param ids        主键值集合
     * @return 结果集
     */
    public JSONArray findAllByIds(String columns, String entityName, JSONArray ids) {
        Map<String, Object> map = META_MAP.get(entityName);
        String idStr = CommonTools.union(ids);
        String sql = "SELECT " + columns + " FROM " + map.get("tableName") + " WHERE " + map.get("idName") + " IN (" + idStr + " )";
        return sqlAction.queryAll("EntityService@findAllByIds",sql);
    }

    public JSONArray findAllByIds(String entityName, JSONArray ids) {
        return findAllByIds("*", entityName, ids);
    }

    /**
     * 查询实体的总数量
     * @param entityName 实体名
     * @return 总数
     */
    public Long getCount(String entityName) {
        Map<String, Object> map = META_MAP.get(entityName);
        String sql = "SELECT COUNT(0) count FROM " + map.get("tableName");
        JSONArray result = sqlAction.queryAll("EntityService@getCount",sql);
        return result.length() > 0 ? result.getJSONObject(0).getLong("count") : 0;
    }

    /**
     * 查询所有数据
     *
     * @param columns    列字段
     * @param entityName 实体名
     * @return 结果集
     */
    public JSONArray findAll(String columns, String entityName) {
        Map<String, Object> map = META_MAP.get(entityName);
        String sql = "SELECT " + columns + " FROM " + map.get("tableName");
        return sqlAction.queryAll("EntityService@findAll",sql);
    }

    public JSONArray findAll(String entityName) {
        return findAll("*", entityName);
    }

    /**
     * 根据ID查询数据
     *
     * @param columns    列字段
     * @param entityName 实体名
     * @param id         主键值
     * @return 实体
     */
    public JSONObject getById(String columns, String entityName, Object id) {
        Map<String, Object> map = META_MAP.get(entityName);
        String sql = "SELECT " + columns + " FROM " + map.get("tableName") + " WHERE " + map.get("idName") + " = '" + id + "'";
        JSONArray result = sqlAction.queryAll("EntityService@getById",sql);
        if (result.length() == 0) {
            return null;
        } else {
            return result.getJSONObject(0);
        }
    }

    public JSONObject getById(String entityName, Object id) {
        return getById("*", entityName, String.valueOf(id));
    }

    /**
     * 实体保存
     *
     * @param entityName 实体名
     * @param row        数据实体
     * @return 结果
     */
    public JSONObject partialSave(String entityName, JSONObject row) throws Exception {
        return partialSave(entityName, new JSONArray().put(row)).getJSONObject(0);
    }

    /**
     * 实体批量保存
     *
     * @param entityName 实体名
     * @param rowArray   数据实体集合
     * @return 结果
     */
    public JSONArray partialSave(String entityName, JSONArray rowArray) throws Exception {
        Map<String, Object> md;
        // lift possible child entity
        if (ENTITY_LIFT_MAP.containsKey(entityName)) {
            String pEntityName = ENTITY_LIFT_MAP.get(entityName);
            md = deepClone(((HashMap<String, Object>) META_MAP.get(pEntityName)));
            enchanceMetaData(md, entityName);
        } else {
            md = META_MAP.get(entityName);
        }

        if (md == null) {
            throw new ServiceException("实体[" + entityName + "]不存在，请检查相关hibernate文件");
        }

        List<Serializable> aid = partialSave(rowArray, md, null);
        JSONArray result = new JSONArray(aid.size());
        aid.forEach(item -> {
            JSONObject obj = new JSONObject();
            obj.put((String) md.get("idName"), item.toString());
            result.put(obj);
        });
        return result;
    }

    private Serializable partialSave(JSONObject row, Map<String, Object> md, Object pIdValue) {
        JSONArray array = new JSONArray(1);
        array.put(row);
        return partialSave(array, md, pIdValue).get(0);
    }

    private List<Serializable> partialSave(JSONArray rowArray, Map<String, Object> md, Object pIdValue) {
        List<Serializable> result = new ArrayList<>(rowArray.length());
        rowArray.forEach(item -> {
            JSONObject row = (JSONObject) item;
            Object idValue = null;
            String idName = (String) md.get("idName");
            String idType = (String) md.get("idType");
            if (row.has(idName)) {
                idValue = row.get(idName);
            }

            @SuppressWarnings("unchecked")
            HashMap<String, Pair> columns = (HashMap<String, Pair>) md.get("columns");
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> links = (HashMap<String, Pair>) md.get("links");

            boolean isInsert = false;
            int affectedRows;

            if (idValue == null) {
                affectedRows = doInsert(row, md, pIdValue, idName, idType, columns);
            } else {
                //if not assigned key or not the ancestor object, and the idvalue is not null, simply update it
                if (!(IDTypeEnum.toType((String) md.get("idGenerator")) == IDTypeEnum.ID_ASSIGNED)) {
                    affectedRows = doUpdate(row, md, idValue, idType, columns);
                } else {
                    if (hasKeyRow((String) md.get("tableName"), (String) md.get("idColName"), idValue, idType)) {
                        affectedRows = doUpdate(row, md, idValue, idType, columns);
                    } else {
                        isInsert = true;
                        affectedRows = doInsert(row, md, pIdValue, idName, idType, columns);
                    }
                }
            }

            //stale object
            if (affectedRows == 0) {
                throw new ServiceException("修改数据失败，返回的影响行数为:0，可能是以下原因之一：1.有其他业务同时修改该条数据导致触发hibernate乐观锁，请检索：hibernate乐观锁。2.update的条件表达式有误");
            }

            //possible modified idValue
            idValue = row.get(idName);


            //remove idName in row
            if (isInsert) {
                row.remove(idName);
            }

            //save or update one-to-many
            Iterator<String> itr = row.keys();
            while (itr.hasNext()) {
                String attr = itr.next();
                if (links.containsKey(attr) && row.has(attr)) {
                    Pair pair = links.get(attr);
                    Map<String, Object> cmd = META_MAP.get(pair.col);
                    if (!row.isNull(attr)) {
                        doSave(row, idValue, attr, cmd);
                    }
                }
            }

            result.add((Serializable) idValue);
        });
        return result;
    }

    private void doSave(JSONObject row, Object idValue, String attr, Map<String, Object> cmd) {
        Object obj = row.get(attr);
        if (obj instanceof JSONArray) {
            JSONArray rows = row.getJSONArray(attr);
            for (int i = 0; i < rows.length(); i++) {
                partialSave(rows.getJSONObject(i), cmd, idValue);
            }
        } else {
            partialSave((JSONObject) obj, cmd, idValue);
        }
    }

    /**
     * hasKeyRow
     */
    private boolean hasKeyRow(String tableName, String idColName, Object idValue, String idType) {
        return sessionPool.getSession().
                createSQLQuery("select 1 from " + tableName + " where " + idColName + "=" + normalizeValue(idValue, idType))
                .list().size() > 0;
    }

    private int doUpdate(JSONObject row, Map<String, Object> md,
                         Object idValue, String idType, HashMap<String, Pair> columns) {
        // 如果只有一个id号，不用更新，返回-1，避免认为更新失败
        if (row.keySet().size() == 1) {
            return -1;
        }
        Object verValue = null;
        String verName = (String) md.get("verName");
        String verType = (String) md.get("verType");
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        StringBuilder sb = new StringBuilder();

        if (verName != null) {
            sb.append(" ").append(verName).append("=").append(normalizeVer(verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sb.append(" ").append(pair.col).append("=").append(row.isNull(attr) ? "null, " : normalizeValue(row.get(attr), pair.type) + ", ");
            }
        }

        if (sb.length() == 0) {
            LOGGER.error("提交的数据错误：" + row);
            throw new RuntimeException("提交的数据错误：" + row);
        }

        sb.delete(sb.length() - 2, sb.length());
        sb.insert(0, "update " + md.get("tableName") + " set ");
        sb.append(" where ").append(md.get("idColName")).append(" = ").append(normalizeValue(idValue, idType));

        if (md.get("verName") != null) {
            sb.append(" and ").append(md.get("verColName")).append("=").append(normalizeValue(verValue, verType));
        }


        LOGGER.info("生成的更新sql: " + sb);
        int affectedRow = SQLUpdate(sb.toString());

        //update possible sub-entity
        if (md.containsKey("subclasses")) {
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> subclasses = (HashMap<String, Pair>) md.get("subclasses");
            for (String sentity : subclasses.keySet()) {
                Pair p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col)) {
                    break;
                }
                //if discriminator matches
                String dv = row.getString(p.col);
                if (p.type.equals(dv) || p.type.contains(dv + ",") || p.type.contains("," + dv)) {
                    StringBuilder buf = new StringBuilder();
                    //update row
                    Map<String, Object> smd = META_MAP.get(sentity);
                    String keyAttr = (String) smd.get("idName");
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> attrs = (HashMap<String, Pair>) smd.get("columns");
                    for (String attr : attrs.keySet()) {
                        //skip key attr
                        if (attr.equals(keyAttr)) {
                            continue;
                        }
                        if (row.has(attr)) {
                            p = attrs.get(attr);
                            buf.append(p.col).append("=").append(normalizeValue(row.isNull(attr) ? null : row.get(attr), p.type)).append(", ");
                        }
                    }

                    buf.delete(buf.length() - 2, buf.length());
                    buf.insert(0, "update " + smd.get("tableName") + " set ");
                    buf.append(" where ").append(smd.get("idColName")).append(" = ").append(normalizeValue(idValue, idType));

                    LOGGER.info("生成的子更新sql: " + buf);
                    SQLUpdate(buf.toString());

                    //继续检查子类的关联，否则就断了链了，但如果父有关联了，儿子就偷懒了
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> links = (HashMap<String, Pair>) smd.get("links");
                    HashMap<String, Pair> pLinks = (HashMap<String, Pair>) md.get("links");
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            Pair pair = links.get(attr);
                            Map<String, Object> cmd = META_MAP.get(pair.col);
                            if (!row.isNull(attr)) {
                                doSave(row, idValue, attr, cmd);
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md);

        return affectedRow;
    }

    private void handleInverses(JSONObject row, Map<String, Object> md) {
        //inverse relation
        @SuppressWarnings("unchecked")
        HashMap<String, String> inverses = (HashMap<String, String>) md.get("inverses");
        for (String inverse : inverses.keySet()) {
            if (row.isNull(inverse)) {
                continue;
            }
            Object arow = row.get(inverse);
            if (arow instanceof JSONObject) {
                Map<String, Object> amd = META_MAP.get(inverses.get(inverse));
                partialSave((JSONObject) arow, amd, null);
            }
        }
    }

    private int doInsert(JSONObject row, Map<String, Object> md,
                         Object pIdValue,
                         String idName, String idType, HashMap<String, Pair> columns) {
        Object verValue = null;
        String verName = (String) md.get("verName");
        String verType = (String) md.get("verType");
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        String idColName = (String) md.get("idColName");

        StringBuilder sbCols = new StringBuilder();
        StringBuilder sbValues = new StringBuilder();

        if (verName != null) {
            sbCols.append(verName).append(", ");
            sbValues.append(normalizeVer(verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sbCols.append(pair.col).append(", ");
                sbValues.append(row.isNull(attr) ? "null, " : normalizeValue(row.get(attr), pair.type) + ", ");
            }
        }


        //handle id problems
        String idStrategy = (String) md.get("idGenerator");
        IDTypeEnum idTypeEnum = IDTypeEnum.toType(idStrategy);
        switch (idTypeEnum) {
            case ID_GUID -> {
                sbCols.append(idColName).append(", ");
                String guid = UUID.randomUUID().toString().replace("-", "");
                sbValues.append("'").append(guid).append("', ");
                row.put(idName, guid);
            }
            case ID_SEQ -> {
                sbCols.append(idColName).append(", ");
                String aid = getLastSeqId((String) md.get("sequence"));
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
            }
            case ID_ASSIGNED -> {
                sbCols.append(idColName).append(", ");
                sbValues.append(normalizeValue(row.get(idName), idType)).append(", ");
                row.put(idName, row.get(idName));
            }
            case ID_FOREIGNER -> {
                sbCols.append(idColName).append(", ");
                String aid = normalizeValue(pIdValue, idType);
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
            }
            default -> {
            }
        }

        if (md.containsKey("inverses")) {
            //handle inverse problems
            @SuppressWarnings("unchecked")
            HashMap<String, String> inverses = (HashMap<String, String>) md.get("inverses");
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> inversesid = (HashMap<String, Pair>) md.get("inverseid");
            for (String inverse : inverses.keySet()) {
                if (row.isNull(inverse)) {
                    continue;
                }
                Object arow = row.get(inverse);
                //LOGGER.debug("发现多对一属性赋值:");
                if (!(arow instanceof JSONObject json)) {
                    Pair pair = inversesid.get(inverse);
                    sbCols.append(pair.col).append(", ");
                    sbValues.append(normalizeValue(arow, pair.type)).append(", ");
                    //LOGGER.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                } else {
                    // 多对一对象关系，属性名与列名必须一致
                    sbCols.append(inverse).append(", ");
                    // 值从给定的对象中取
                    Pair pair = inversesid.get(inverse);
                    arow = json.get(pair.col);
                    sbValues.append(normalizeValue(arow, pair.type)).append(", ");
                    //LOGGER.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                }
            }
        }

        sbCols.delete(sbCols.length() - 2, sbCols.length());
        sbCols.insert(0, "insert into " + md.get("tableName") + " (");
        sbValues.delete(sbValues.length() - 2, sbValues.length());
        sbCols.append(") values (");
        sbCols.append(sbValues);
        sbCols.append(")");

        LOGGER.info("生成的插入sql；" + sbCols);

        int affectedRows;

        if (idTypeEnum == IDTypeEnum.ID_AUTO) {
            Pair pair;
            try {
                pair = rawJdbcUpdate(sbCols.toString());
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            row.put(idName, pair.type);
            affectedRows = Integer.parseInt(pair.col);
        } else {
            affectedRows = SQLUpdate(sbCols.toString());
        }

        //insert possible sub-entity
        if (md.containsKey("subclasses")) {
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> subclasses = (HashMap<String, Pair>) md.get("subclasses");
            for (String sentity : subclasses.keySet()) {
                Pair p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col)) {
                    break;
                }
                String dv = row.getString(p.col);
                if (p.type.equals(dv) || p.type.contains(dv + ",") || p.type.contains("," + dv)) {
                    StringBuilder fields = new StringBuilder();
                    StringBuilder vals = new StringBuilder();

                    //update row
                    Map<String, Object> smd = META_MAP.get(sentity);
                    String keyCol = (String) smd.get("idColName");

                    //handle it
                    fields.append(keyCol).append(", ");
                    vals.append(normalizeValue(row.get(idName), idType)).append(", ");

                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> attrs = (HashMap<String, Pair>) smd.get("columns");
                    for (String attr : attrs.keySet()) {
                        if (row.has(attr)) {
                            p = attrs.get(attr);
                            // 外键不当属性加入
                            fields.append(p.col).append(", ");
                            vals.append(normalizeValue(row.isNull(attr) ? null : row.get(attr), p.type)).append(", ");
                        }
                    }

                    //处理反向查id的情况
                    if (smd.containsKey("inverseid")) {
                        @SuppressWarnings("unchecked")
                        HashMap<String, Pair> lookup = (HashMap<String, Pair>) smd.get("inverseid");
                        for (String property : lookup.keySet()) {
                            if (row.has(property) && !row.isNull(property)) {
                                Object obj = row.get(property);
                                Pair pair = lookup.get(property);
                                if (obj instanceof JSONObject record) {
                                    if (record.has(pair.col) && !record.isNull(pair.col)) {
                                        fields.append(pair.col).append(", ");
                                        vals.append(normalizeValue(record.get(pair.col), pair.type)).append(", ");
                                    }
                                } else {
                                    fields.append(pair.col).append(", ");
                                    vals.append(normalizeValue(row.get(property), pair.type)).append(", ");
                                }
                            }
                        }
                    }

                    fields.delete(fields.length() - 2, fields.length());
                    vals.delete(vals.length() - 2, vals.length());

                    fields.insert(0, "insert into " + smd.get("tableName") + " (");
                    fields.append(") values (");
                    fields.append(vals);
                    fields.append(")");

                    LOGGER.info("生成的子插入sql: " + fields);
                    SQLUpdate(fields.toString());

                    //继续检查子类的关联，否则就断了链了
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> links = (HashMap<String, Pair>) smd.get("links");
                    HashMap<String, Pair> pLinks = (HashMap<String, Pair>) md.get("links");
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            Pair pair = links.get(attr);
                            Map<String, Object> cmd = META_MAP.get(pair.col);
                            if (!row.isNull(attr)) {
                                Object obj = row.get(attr);
                                if (obj instanceof JSONArray) {
                                    JSONArray rows = row.getJSONArray(attr);
                                    for (int i = 0; i < rows.length(); i++) {
                                        partialSave(rows.getJSONObject(i), cmd, row.get(idName));
                                    }
                                } else {
                                    partialSave((JSONObject) obj, cmd, row.get(idName));
                                }
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md);

        return affectedRows;
    }

    private Pair rawJdbcUpdate(String sql) throws Exception {
        Pair pair = new Pair("", "");
        Connection con = ((SessionImpl) sessionPool.getSession()).connection();
        PreparedStatement statement = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
        ResultSet rs = null;
        try {
            int affectedRows = statement.executeUpdate();
            pair.col = affectedRows + "";
            if (affectedRows == 0) {
                return pair;
            } else {
                rs = statement.getGeneratedKeys();
                if (rs.next()) {
                    pair.type = rs.getLong(1) + "";
                } else {
                    throw new Exception("得到插入记录id失败。");
                }
            }
            return pair;
        } finally {
            if (rs != null) {
                rs.close();
            }
            if (statement != null) {
                statement.close();
            }
        }
    }

    private int SQLUpdate(String sql) {
        NativeQuery<?> query = sessionPool.getSession().createNativeQuery(sql);
        return query.executeUpdate();
    }

    private String query(String sql) {
        NativeQuery<?> query = sessionPool.getSession().createNativeQuery(sql);
        return query.list().get(0) + "";
    }

    private String getLastSeqId(String seq) {
        return query("select " + seq + ".nextval newid from dual");
    }

    private String normalizeVer(Object verValue, String verType) {
        DialectTypeEnum dialectType = getDialectType();
        if (ColumnTypeEnum.toType(verType) == ColumnTypeEnum.COL_TIME) {
            if (dialectType == DialectTypeEnum.DIALECT_ORACLE) {
                return "current_timestamp";
            } else {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                return "'" + sdf.format(new Date()) + "'";
            }
        } else if (verValue == null) {
            return "1";
        } else {
            return (Integer.parseInt(verValue + "") + 1) + "";
        }
    }

    /**
     * normalizeValue
     */
    private String normalizeValue(Object value, String valType) {
        if (value == null) {
            return "null";
        }
        ColumnTypeEnum columnTypeEnum = ColumnTypeEnum.toType(valType);
        DialectTypeEnum dialectType = getDialectType();
        switch (columnTypeEnum) {
            case COL_STRING:
                return "'" + value + "'";
            case COL_NUMBER:
                if ("".equals(value)) {
                    return "null";
                }
                return value + "";
            case COL_BOOLEAN:
                if (dialectType == DialectTypeEnum.DIALECT_ORACLE) {
                    return "'" + ((Boolean) value ? "Y" : "N") + "'";
                } else {
                    //该if是为了解决在SQLServer下保存数据时可能会导致Byte转Boolean时失败
                    if (value instanceof Byte) {
                        return value.toString();
                    } else if (value instanceof Integer) {
                        return value.toString();
                    } else {
                        return (Boolean) value ? "1" : "0";
                    }
                }
            case COL_DATE:
                if (dialectType == DialectTypeEnum.DIALECT_ORACLE) {
                    return "TO_DATE(SUBSTR('" + value + "', 1, 10), 'YYYY-MM-DD')";
                } else {
                    return "'" + value + "'";
                }
            case COL_TIME:
                if (dialectType == DialectTypeEnum.DIALECT_ORACLE) {
                    return "TO_TIMESTAMP('" + value + "', 'YYYY-MM-DD HH24:MI:SS')";
                } else {
                    return "'" + value + "'";
                }
            default:
                throw new ServiceException("传递数据类型出错");
        }
    }

    /**
     * preload all meta data
     */
    public void loadMetaData() {
        //获取所有实体元数据
        MetamodelImplementor metaModelImpl = sessionPool.getSessionFactory().getMetamodel();
        Map<String, EntityPersister> entityPersisterMap = metaModelImpl.entityPersisters();
        Collection<EntityPersister> val = entityPersisterMap.values();
        Map<String, ClassMetadata> map = new LinkedHashMap<>(val.size());
        for (EntityPersister ep : val) {
            AbstractEntityPersister aep = (AbstractEntityPersister) ep;
            map.put(aep.getEntityName(), aep);
        }
        //获取每个实体的元数据
        for (String entityName : map.keySet()) {
            try {
                this.getMetaData(entityName);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        //跟踪实体的继承关系
        SAXReader reader = new SAXReader();
        // 加载根目录下的subclass
        IOTools.getStream("subclass.xml",
                stream -> loadSubclass(reader, stream));

        // 加载模块下的subclass
        // 解析subclass.xml文件 获取模块
        Map<String, Map<String, String>> maps = EntityService.getSubClassMap();
        for (String moduleName : maps.keySet()) {
            // module要添加路径分隔符
            if (!"".equals(moduleName)) {
                moduleName += "/";
            }

            String finalModuleName = moduleName;
            IOTools.getStream("/" + finalModuleName + "subclass.xml",
                    stream -> loadSubclass(reader, stream),
                    notFindPath -> LOGGER.info(finalModuleName + "模块下无subclass.xml文件"));
        }

        //上面处理的都是正向关系，下面处理反向多对一关系
        //find out inverse many to one relation
        for (String entityName : map.keySet()) {
            try {
                Map<String, Object> hash = META_MAP.get(entityName);
                if (hash != null) {
                    ClassMetadata cmd = sessionPool.getSessionFactory().getClassMetadata(entityName);
                    HashMap<String, String> inverses = new HashMap<>(16);
                    hash.put("inverses", inverses);
                    HashMap<String, Pair> inverseMap = new HashMap<>(16);
                    hash.put("inverseid", inverseMap);
                    for (String property : cmd.getPropertyNames()) {
                        Type type = cmd.getPropertyType(property);
                        //handle inverse relationship
                        if (type instanceof ManyToOneType mto) {
                            String linkedEntity = mto.getAssociatedEntityName();
                            String colName = mto.getAssociatedJoinable(sessionPool.getSessionFactory()).getKeyColumnNames()[0];
                            inverses.put(property, linkedEntity);
                            inverseMap.put(property, new Pair(colName, (String) META_MAP.get(linkedEntity).get("idType")));
                        }
                    }
                } else {
                    LOGGER.warn("未处理【" + entityName + "】的反向多对一关系，可能含有不支持的映射结构");
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        //LOGGER.debug("meta data all loaded.");
    }

    private void loadSubclass(SAXReader reader, InputStream input) {
        try {
            Document document = reader.read(input);
            Element root = document.getRootElement();
            for (Iterator<Element> it = root.elementIterator("entity"); it.hasNext(); ) {
                Element elm = it.next();
                String entityName = elm.attribute("name").getValue();
                String parentEntity = elm.attribute("parentEntity").getValue();
                String discProperty = elm.attribute("discProperty").getValue();
                String discriminator = elm.attribute("discriminator").getValue();
                if (META_MAP.containsKey(parentEntity)) {
                    Map<String, Object> em = META_MAP.get(parentEntity);
                    HashMap<String, Pair> subclasses;
                    if (em.containsKey("subclasses")) {
                        subclasses = (HashMap<String, Pair>) em.get("subclasses");
                    } else {
                        subclasses = new HashMap<>(1);
                        em.put("subclasses", subclasses);
                    }
                    subclasses.put(entityName, new Pair(discProperty, discriminator));
                    ENTITY_LIFT_MAP.put(entityName, parentEntity);
                }
            }
        } catch (DocumentException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 获取列类型
     */
    private String normalizeType(String type) {
        type = type.toLowerCase();
        return switch (type) {
            case "string" -> ColumnTypeEnum.COL_STRING.getValue();
            case "date" -> ColumnTypeEnum.COL_DATE.getValue();
            case "time", "timestamp" -> ColumnTypeEnum.COL_TIME.getValue();
            case "boolean", "yes_no" -> ColumnTypeEnum.COL_BOOLEAN.getValue();
            case "integer", "double", "big_decimal", "big_integer" -> ColumnTypeEnum.COL_NUMBER.getValue();
            default -> ColumnTypeEnum.COL_NOT_SUPPORTED.getValue();
        };
    }

    /**
     * 获取元数据
     *
     * @param entityName 实体名
     */
    private void getMetaData(String entityName) throws Exception {
        // 如果元数据已经获取，直接返回
        if (META_MAP.containsKey(entityName)) {
            return;
        }

        Map<String, Object> map = new HashMap<>(12);

        //实体名称
        map.put("entityName", entityName);

        //获取表名
        ClassMetadata cmd = sessionPool.getSessionFactory().getClassMetadata(entityName);
        AbstractEntityPersister aep = (AbstractEntityPersister) cmd;
        map.put("tableName", aep.getTableName());

        //获取主键属性名称
        String attr = cmd.getIdentifierPropertyName();
        if (attr == null) {
            LOGGER.warn("停止处理【" + entityName + "】的映射，可能含有不支持的映射结构");
            return;
        }
        map.put("idName", attr);
        map.put("idColName", aep.getPropertyColumnNames(attr)[0]);
        //获取主键类型
        String idType = normalizeType(cmd.getIdentifierType().getName());
        map.put("idType", idType);

        //仅支持guid、sequence（oracle）、auto-increment、assigned、outside（id来自主实体的一对一）
        IdentifierGenerator idg = aep.getIdentifierGenerator();
        if (idg instanceof GUIDGenerator || idg instanceof UUIDGenerator
                || idg instanceof UUIDHexGenerator) {
            map.put("idGenerator", IDTypeEnum.ID_GUID.getValue());
        } else if (idg instanceof SequenceStyleGenerator generator) {
            String name = generator.getDatabaseStructure().getPhysicalName().render();
            // 创建sequence，hibernate没有自动创建 (被MyOracle10gDialect代替)
            // this.createSequence(name);
            map.put("sequence", name);
            map.put("idGenerator", IDTypeEnum.ID_SEQ.getValue());
        } else if (idg instanceof IdentityGenerator) {
            map.put("idGenerator", IDTypeEnum.ID_AUTO.getValue());
        } else if (idg instanceof Assigned) {
            map.put("idGenerator", IDTypeEnum.ID_ASSIGNED.getValue());
        } else if (idg instanceof ForeignGenerator) {
            map.put("idGenerator", IDTypeEnum.ID_FOREIGNER.getValue());
        } else {
            throw new Exception("Unsupported id generator strategy:" + idg.getClass().getName());
        }

        //判断该实体是否有乐观锁（即版本号）控制
        if (cmd.isVersioned()) {
            attr = cmd.getPropertyNames()[cmd.getVersionProperty()];
            map.put("verName", attr);
            map.put("verColName", aep.getPropertyColumnNames(attr)[0]);
            map.put("verType", normalizeType(cmd.getPropertyType(attr).getName()));
        }

        //普通列
        HashMap<String, Pair> columns = new HashMap<>(1);
        map.put("columns", columns);

        //一对一，一对多关系映射
        HashMap<String, Pair> links = new HashMap<>(1);
        map.put("links", links);

        //一对一
        HashMap<String, Pair> onetoone = new HashMap<>(1);
        map.put("onetoone", onetoone);

        //一对多
        HashMap<String, Pair> onetomany = new HashMap<>(1);
        map.put("onetomany", onetomany);

        //遍历实体属性集合
        for (String property : cmd.getPropertyNames()) {
            //获取类型
            Type type = cmd.getPropertyType(property);
            if (type instanceof CollectionType || type instanceof OneToOneType) {
                org.hibernate.persister.entity.Joinable ja;
                String linkEntity;
                SessionFactoryImplementor sf = sessionPool.getSessionFactory();
                if (type instanceof CollectionType st) {
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                } else {
                    OneToOneType st = (OneToOneType) type;
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                }
                String foreignKey = ja.getKeyColumnNames()[0];
                String link;
                if (type instanceof OneToOneType) {
                    link = property;
                } else {
                    link = ja.getName().substring(entityName.length() + 1);
                }
                links.put(link, new Pair(linkEntity, foreignKey));
                // 把一对多和一对一分开
                if (type instanceof CollectionType) {
                    onetomany.put(link, new Pair(linkEntity, foreignKey));
                } else {
                    onetoone.put(link, new Pair(linkEntity, foreignKey));
                }
            } else if (!(type instanceof ManyToOneType)) {
                String columnName = ((AbstractEntityPersister) cmd).getPropertyColumnNames(property)[0];
                columns.put(property, new Pair(columnName, normalizeType(type.getName())));
            }
        }
        META_MAP.put(entityName, map);
    }

    /**
     * 深克隆
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> T deepClone(T o) throws Exception {
        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(byteOut);
        out.writeObject(o);
        out.flush();
        ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(byteOut.toByteArray()));
        return (T) o.getClass().cast(in.readObject());
    }

    /**
     * 解析元数据
     */
    @SuppressWarnings("unchecked")
    private void enchanceMetaData(Map<String, Object> pmd, String entityName) throws Exception {
        Map<String, Object> md = META_MAP.get(entityName);

        HashMap<String, Pair> links = (HashMap<String, Pair>) md.get("links");
        HashMap<String, Pair> plinks = (HashMap<String, Pair>) pmd.get("links");
        for (String key : links.keySet()) {
            plinks.put(key, deepClone(links.get(key)));
        }

        HashMap<String, Pair> sub;
        if (md.containsKey("subclasses")) {
            sub = (HashMap<String, Pair>) md.get("subclasses");
        } else {
            sub = new HashMap<>(0);
        }
        HashMap<String, Pair> psub;
        if (pmd.containsKey("subclasses")) {
            psub = (HashMap<String, Pair>) pmd.get("subclasses");
        } else {
            psub = new HashMap<>(sub.size());
        }
        for (String key : sub.keySet()) {
            psub.put(key, deepClone(sub.get(key)));
        }

        HashMap<String, String> inv = (HashMap<String, String>) md.get("inverses");
        HashMap<String, String> pinv = (HashMap<String, String>) pmd.get("inverses");
        for (String key : inv.keySet()) {
            pinv.put(key, inv.get(key));
        }

        HashMap<String, Pair> invid;
        if (md.containsKey("inverseid")) {
            invid = (HashMap<String, Pair>) md.get("inverseid");
        } else {
            invid = new HashMap<>(0);
        }
        HashMap<String, Pair> pinvid;
        if (pmd.containsKey("inverseid")) {
            pinvid = (HashMap<String, Pair>) pmd.get("inverseid");
        } else {
            pinvid = new HashMap<>(invid.size());
        }

        for (String key : invid.keySet()) {
            pinvid.put(key, deepClone(invid.get(key)));
        }
    }
}
