package com.af.v4.system.restful.config;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.plugins.io.IOTools;
import com.af.v4.system.plugins.json.JsonTools;
import com.af.v4.system.restful.dialect.MySqlServerDialect;
import com.af.v4.system.restful.exception.FileNotFoundException;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * 系统配置
 *
 * @author Mr.river
 */
@Component
public final class SystemConfig {
    private final RedisService redisService;
    public static final String CONFIG_PATH = "systemConfig.json";

    public static final String STANDARD_VERSION_VALUE = "standard";

    public static final String VERSION_ALIAS_KEY = "System@versionAlias";

    private static final Logger LOGGER = LoggerFactory.getLogger(SystemConfig.class);
    private static boolean isLoaded = false;

    public SystemConfig(RedisService redisService) {
        this.redisService = redisService;
    }

    /**
     * 获取系统定制版本别名
     * @return 系统定制版本别名
     */
    public String getVersionAliasValue() {
        String versionAliasValue = redisService.get(VERSION_ALIAS_KEY);
        if(versionAliasValue == null){
            LOGGER.info("等待af-system设置系统定制版本别名...");
            try {
                Thread.sleep(1500);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            versionAliasValue = getVersionAliasValue();
        }
        return versionAliasValue;
    }

    /**
     * 获取配置文件并加载
     */
    private static void load() {
        //获取配置文件
        if (!isLoaded) {
            isLoaded = true;
            if (!IOTools.hasResource(CONFIG_PATH)) {
                throw new FileNotFoundException("系统配置文件" + CONFIG_PATH + "不存在", CONFIG_PATH);
            }
            JSONObject params = JsonTools.readJsonFile(CONFIG_PATH);
            /* 系统配置 **/
            JSONObject system = params.getJSONObject("system");
            if (!system.has("webAppName")) {
                throw new ServiceException(CONFIG_PATH + "中服务应用模块名参数[system:webAppName]不存在，请配置该参数为web应用名称，如revenue,webmeter等");
            }
            System.webAppName = system.getString("webAppName");
            /* 资源类配置 **/
            JSONObject resources = params.getJSONObject("resources");
            if (!resources.has("fileRootPath")) {
                LOGGER.warn(CONFIG_PATH + "中资源根目录参数[resources:fileRootPath]不存在，以按照缺省值[C:/" + System.webAppName + "Res]配置");
            }
            Resources.fileRootPath = resources.optString("fileRootPath", "C:/" + System.webAppName + "Res");
            //  如果有扩展目录配置，则追加
            if (resources.has("extraPaths")) {
                Resources.extraPaths = resources.getJSONObject("extraPaths");
            }
            /* Hibernate配置 **/
            JSONObject hibernate = params.getJSONObject("hibernate");
            if (!hibernate.has("cfgLocations")) {
                throw new ServiceException(CONFIG_PATH + "中Hibernate实体配置参数[hibernate:cfgLocations]不存在");
            }
            // 方言类配置
            Hibernate.dialect = hibernate.optString("dialect", MySqlServerDialect.class.getCanonicalName());
            // 实体映射配置
            Hibernate.cfgLocations = hibernate.getJSONObject("cfgLocations");
            // 实体关联映射配置
            Hibernate.subClassArray = hibernate.getJSONArray("subclass");
        }
    }

    /**
     * 系统
     */
    public static class System {

        private static String webAppName;

        static {
            SystemConfig.load();
        }

        public static String getWebAppName() {
            return webAppName;
        }
    }

    /**
     * 资源
     */
    public static class Resources {
        private static String fileRootPath;
        private static JSONObject extraPaths;

        static {
            SystemConfig.load();
        }

        public static String getFileRootPath() {
            return fileRootPath;
        }

        public static String getExtraPath(String key) {
            return extraPaths.getString(key);
        }
    }

    /**
     * hibernate配置
     */
    public static class Hibernate {
        private static String dialect;
        private static JSONObject cfgLocations;
        private static JSONArray subClassArray;

        static {
            SystemConfig.load();
        }

        public static String getDialect() {
            return dialect;
        }

        public static JSONObject getCfgLocations() {
            return cfgLocations;
        }

        public static JSONArray getSubClassArray() {
            return subClassArray;
        }

        public static boolean hasSubClassArray() {
            return subClassArray != null;
        }
    }
}
