package com.af.v4.system.restful.config;

import com.af.v4.system.restful.sql.dynamic.DynamicDataSource;
import com.alibaba.druid.pool.DruidAbstractDataSource;
import com.alibaba.druid.pool.DruidDataSource;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.ConstructorBinding;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;

import java.sql.SQLException;
import java.util.*;

@Configuration
@ConfigurationProperties(prefix = "db")
@Component
public class DynamicDataSourceConfig {

    private List<DruidDataSource> dataSourceList;

    private Integer initialSize;

    private Integer minIdle;

    private Integer maxActive;

    private Integer maxWait;

    @Bean
    @Primary
    public DynamicDataSource routingDataSource() {
        // 循环加入数据源，实例化动态数据源工具类
        Map<Object, Object> targetDataSources = new HashMap<>(dataSourceList.size());
        dataSourceList.forEach(item -> {
            setProperties(item);
            targetDataSources.put(item.getName(), item);
        });
        return new DynamicDataSource(dataSourceList.get(0), targetDataSources);
    }

    private void setProperties(DruidDataSource item){
        if(item.getInitialSize() == DruidAbstractDataSource.DEFAULT_INITIAL_SIZE){
            item.setInitialSize(Objects.requireNonNullElse(initialSize, 5));
        }
        if(item.getMinIdle() == DruidAbstractDataSource.DEFAULT_MIN_IDLE){
            item.setMinIdle(Objects.requireNonNullElse(minIdle, 5));
        }
        if(item.getMaxActive() == DruidAbstractDataSource.DEFAULT_MAX_ACTIVE_SIZE){
            item.setMaxActive(Objects.requireNonNullElse(maxActive, 64));
        }
        if(item.getMaxWait() == DruidAbstractDataSource.DEFAULT_MAX_WAIT){
            item.setMaxWait(Objects.requireNonNullElse(maxWait, 60000));
        }
        item.setTimeBetweenEvictionRunsMillis(60000);
        item.setMinEvictableIdleTimeMillis(1800000);
        item.setTestWhileIdle(true);
        item.setTestOnBorrow(false);
        item.setTestOnReturn(false);
        item.setUseUnfairLock(true);
        String driverClassName = item.getDriverClassName();
        if(driverClassName.contains("SQLServer") || driverClassName.contains("Oracle")){
            item.setPoolPreparedStatements(true);
            item.setMaxPoolPreparedStatementPerConnectionSize(20);
            if(driverClassName.contains("SQLServer")){
                item.setValidationQuery("select 1");
            }
        }
        if(Objects.equals(item.getValidationQuery(), DruidAbstractDataSource.DEFAULT_VALIDATION_QUERY)){
            item.setValidationQuery("select 1 from dual");
        }
        try {
            item.setFilters("stat,slf4j");
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        Properties properties = new Properties();
        properties.setProperty("druid.stat.mergeSql", "true");
        properties.setProperty("druid.stat.slowSqlMilli", "5000");
        item.setConnectProperties(properties);
    }

    public List<DruidDataSource> getDataSourceList() {
        return dataSourceList;
    }

    public void setDataSourceList(List<DruidDataSource> dataSourceList) {
        this.dataSourceList = dataSourceList;
    }

    public Integer getInitialSize() {
        return initialSize;
    }

    public void setInitialSize(Integer initialSize) {
        this.initialSize = initialSize;
    }

    public Integer getMinIdle() {
        return minIdle;
    }

    public void setMinIdle(Integer minIdle) {
        this.minIdle = minIdle;
    }

    public Integer getMaxActive() {
        return maxActive;
    }

    public void setMaxActive(Integer maxActive) {
        this.maxActive = maxActive;
    }

    public Integer getMaxWait() {
        return maxWait;
    }

    public void setMaxWait(Integer maxWait) {
        this.maxWait = maxWait;
    }
}
