package com.af.v4.system.restful.controller;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * Entity控制器
 *
 * @author Mr.river
 */
@RestController
@RequestMapping("/entity")
public class EntityController {
    private final EntityService entityService;

    public EntityController(EntityService entityService) {
        this.entityService = entityService;
    }

    @Log(title = "保存/更新实体", businessType = BusinessType.Entity)
    @PostMapping(value = "/save/{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Map<String, Object>> save(@PathVariable("entityName") String entityName,
                                       @RequestBody String entityContent) throws Exception {
        return R.ok(entityService.partialSave(entityName, new JSONObject(entityContent)).toMap());
    }

    @Log(title = "批量保存/更新实体", businessType = BusinessType.Entity)
    @PostMapping(value = "/batchSave/{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<List<Object>> saveBatch(@PathVariable("entityName") String entityName,
                                     @RequestBody String entityContent) throws Exception {
        return R.ok(entityService.partialSave(entityName, new JSONArray(entityContent)).toList());
    }


    @Log(title = "删除实体", businessType = BusinessType.Entity)
    @DeleteMapping(value = "{entityName}/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Map<String, Object>> deleteById(@PathVariable("entityName") String entityName,
                                             @PathVariable("id") String id) {
        JSONObject result = new JSONObject();
        result.put("msg", entityService.delete(entityName, id));
        return R.ok(result.toMap());
    }


    @Log(title = "批量删除实体", businessType = BusinessType.Entity)
    @DeleteMapping(value = "{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Map<String, Object>> deleteAllByIds(@PathVariable("entityName") String entityName,
                                                 @RequestBody String entity) {
        JSONObject entityContent = new JSONObject(entity);
        JSONObject result = new JSONObject();
        result.put("msg", entityService.deleteAllByIds(entityName, entityContent.getJSONArray("ids")));
        return R.ok(result.toMap());
    }

    @Log(title = "根据ID查询数据", businessType = BusinessType.Entity)
    @PostMapping(value = "/query/{entityName}/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Map<String, Object>> getById(@PathVariable("entityName") String entityName,
                                          @PathVariable("id") String id,
                                          @RequestBody String entity) {
        JSONObject entityContent = new JSONObject(entity);
        if(entityContent.has("columns")){
            return R.ok(entityService.getById(entityContent.getString("columns"), entityName, id).toMap());
        } else {
            return R.ok(entityService.getById(entityName, id).toMap());
        }
    }

    @Log(title = "根据ID集合查询所有数据", businessType = BusinessType.Entity)
    @PostMapping(value = "/query/{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<List<Object>> findAllByIds(@PathVariable("entityName") String entityName,
                                        @RequestBody String entity) {
        JSONObject entityContent = new JSONObject(entity);
        if(entityContent.has("columns")){
            return R.ok(entityService.findAllByIds(entityContent.getString("columns"), entityName, entityContent.getJSONArray("ids")).toList());
        } else {
            return R.ok(entityService.findAllByIds(entityName, entityContent.getJSONArray("ids")).toList());
        }
    }

    @Log(title = "查询所有数据", businessType = BusinessType.Entity)
    @PostMapping(value = "/queryAll/{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<List<Object>> findAll(@PathVariable("entityName") String entityName,
                                        @RequestBody String entity) {
        JSONObject entityContent = new JSONObject(entity);
        if(entityContent.has("columns")){
            return R.ok(entityService.findAll(entityContent.getString("columns"), entityName).toList());
        } else {
            return R.ok(entityService.findAll(entityName).toList());
        }
    }

    @Log(title = "查询实体的总数量", businessType = BusinessType.Entity)
    @PostMapping(value = "/queryCount/{entityName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Map<String, Object>> getCount(@PathVariable("entityName") String entityName) {
        JSONObject result = new JSONObject();
        result.put("msg", entityService.getCount(entityName));
        return R.ok(result.toMap());
    }
}
