package com.af.v4.system.restful.sql;

import com.af.v4.system.restful.transaction.SessionPool;
import org.hibernate.Session;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.query.NativeQuery;
import org.hibernate.transform.ResultTransformer;
import org.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Component;

/**
 * SQL执行类
 *
 * @author Mr.river
 */
@Component
public class SqlAction {

    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    private final SessionPool sessionPool;

    public SqlAction(SessionPool sessionPool) {
        this.sessionPool = sessionPool;
    }

    public JSONArray query(String name, String sql) {
        return query(name, sql, 0, 1000);
    }

    public JSONArray query(String name, String sql, int pageNo,
                           int pageSize) {
        return (JSONArray) sqlMonitor(() -> {
            HibernateSQLCall sqlCall = new HibernateSQLCall(sql, pageNo, pageSize);
            sqlCall.transformer = AliasTransformer.INSTANCE;
            return sqlCall.doInHibernate(sessionPool.getSession());
        }, name, sql);
    }

    public int bulkSQLUpdate(String name, String sql) {
        return (int) sqlMonitor(() -> {
            Session session = sessionPool.getSession();
            NativeQuery<?> queryObject = session.createNativeQuery(sql);
            return queryObject.executeUpdate();
        }, name, sql);
    }

    private Object sqlMonitor(Process process, String name, String sql) {
        long begin = System.currentTimeMillis();
        try {
            return process.apply();
        } finally {
            long end = System.currentTimeMillis();
            LOGGER.debug("维护信息:执行SQL["+name+"]耗时：" + (end - begin) + "ms:{\n" + sql + "\n}");
        }
    }

    @FunctionalInterface
    interface Process {
        Object apply();
    }

    private static class HibernateSQLCall implements HibernateCallback<JSONArray> {
        final String sql;
        final int page;
        final int rows;
        public ResultTransformer transformer = null;

        public HibernateSQLCall(String sql, int page, int rows) {
            this.sql = sql;
            this.page = page;
            this.rows = rows;
        }

        @Override
        public JSONArray doInHibernate(Session session) {
            NativeQuery<?> q = session.createNativeQuery(sql);
            if (transformer != null) {
                q.setResultTransformer(transformer);
            }
            try {
                return new JSONArray(q.setFirstResult(page * rows).setMaxResults(rows).list());
            } catch (SQLGrammarException ex) {
                // 把sql语句添加到异常信息中
                String msg = "sql:\n" + sql + "\n" + ex.getMessage();
                throw new SQLGrammarException(msg, ex.getSQLException());
            }
        }
    }
}
