package com.af.v4.system.restful.mapper;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.plugins.io.IOTools;
import com.af.v4.system.restful.utils.SpringBeansPour;
import com.af.v4.system.restful.utils.ThreadResource;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RestController;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * 抽象资源映射器
 */
@Component
public abstract class AbstractResourceMapper<T> implements ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractResourceMapper.class);
    private static ApplicationContext applicationContext;
    private final ModuleMapper moduleMapper;
    /**
     * 资源列表
     */
    protected Map<String, Map<String, T>> map;

    protected AbstractResourceMapper(ModuleMapper moduleMapper) {
        this.moduleMapper = moduleMapper;
    }

    /**
     * 获取资源类型
     *
     * @return 资源类型
     */
    protected abstract String getResType();

    /**
     * 获取资源文件名
     *
     * @return 资源文件名
     */
    protected abstract String getFileName();

    /**
     * 获取资源文件夹名
     *
     * @return 资源文件夹名
     */
    protected abstract String getFolderName();

    /**
     * 加载资源列表
     */
    protected void loadMap() {
        if (map == null) {
            map = new HashMap<>(36);
            putMapByRoot();
            moduleMapper.getMap().forEach((key, value) -> {
                String name = value.get("name");
                String path = value.get("path");
                try {
                    putMapByModule(name, path);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
    }

    /**
     * 加载根目录资源到列表
     */
    protected void putMapByRoot() {
        putMapByModule(null, null);
    }

    /**
     * 加载模块资源到列表
     *
     * @param name   模块名
     * @param parent 父模块名
     */
    protected void putMapByModule(String name, String parent) {
        String str;
        String fileName = getFileName();
        String resType = getResType();
        if (parent == null) {
            if (name == null) {
                str = fileName;
            } else {
                str = name + "/" + fileName;
            }
        } else {
            str = parent + name + "/" + fileName;
        }
        IOTools.getStream(str, stream -> {
            SAXReader reader = new SAXReader();
            Document docLogic;
            try {
                docLogic = reader.read(stream);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
            Element rootLogic = docLogic.getRootElement();
            for (Iterator<Element> item = rootLogic.elementIterator(resType); item.hasNext(); ) {
                Element element = item.next();
                appendRes(element, name, parent);
            }
        });
    }

    /**
     * 获取资源Map
     *
     * @return 全部资源
     */
    public Map<String, Map<String, T>> getAllMap() {
        return map;
    }

    /**
     * 根据Key获取值
     *
     * @param key KEY
     * @return value
     */
    public T getValue(String key) {
        if (map.containsKey(key)) {
            T value = map.get(key).get("value");
            if (value instanceof String) {
                T result = getValueByUser(key);
                if (result != null) {
                    return result;
                }
            }
            return value;
        } else {
            return null;
        }
    }

    /**
     * 根据Key获取对应资源Map
     * @param key KEY
     * @return 资源Map
     */
    public Map<String, T> getRes(String key){
        return map.getOrDefault(key, null);
    }

    public T getValueByUser(String alias) {
        String folderName = getFolderName();
        if (folderName == null) {
            return null;
        }
        //获取登陆用户用户名
        String dir = ThreadResource.ComponentDir.get();
        if (dir == null) {
            return null;
        }
        //查看对应目录下是否存在相关资源
        String path = "/" + dir + "/" + folderName + "/" + alias + "." + getResType();
        if (IOTools.hasResource(path)) {
            return (T) path;
        }
        return null;
    }

    /**
     * 追加资源
     *
     * @param element          XML元素
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    protected Map<String, T> appendRes(Element element, String moduleName, String parentModuleName) {
        String alias = element.attribute("alias").getValue();

        // 验证别名是否重复
        if (map.containsKey(alias)) {
            String resType = getResType();
            throw new ServiceException(resType + "别名" + alias + "已存在");
        }
        // 设置属性
        Map<String, T> res = new HashMap<>(4);
        T value;
        Attribute pathAttribute = element.attribute("path");
        if (pathAttribute != null) {
            String path = pathAttribute.getValue();
            res.put("alias", (T) alias);
            String folderName = getFolderName();
            path = (parentModuleName == null ?
                    moduleName + "/" + folderName + "/" :
                    parentModuleName + "/" + moduleName + "/" + folderName + "/") + path;
            value = (T) path;
        } else {
            String className = element.attribute("class").getValue();
            Object obj = getClassByName(className, moduleName);
            value = (T) obj;
        }
        res.put("value", value);
        map.put(alias, res);
        return res;
    }

    private Object getClassByName(String className, String moduleName) {
        try {
            Class<?> c = Class.forName(className);
            Object obj;
            // 如果是spring的bean，让spring取
            if (isSpringBean(c)) {
                obj = applicationContext.getBean(c);
            } else {
                obj = c.getDeclaredConstructor().newInstance();
                // 如果实现了插件Bean注入工具
                if (obj instanceof SpringBeansPour) {
                    ((SpringBeansPour) obj).initBeans(applicationContext);
                }
            }
            return obj;
        } catch (ClassNotFoundException | NoClassDefFoundError ex) {
            LOGGER.warn("未找到类, 模块=" + moduleName + ", 类=" + ex.getMessage());
            return null;
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    private boolean isSpringBean(Class<?> c) {
        for (Annotation annotation : c.getAnnotations()) {
            if (annotation.annotationType() == Component.class ||
                    annotation.annotationType() == Service.class ||
                    annotation.annotationType() == RestController.class
            ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Spring启动的时候会自动调用下面的方法设置ApplicationContext的值
     */
    @Override
    public void setApplicationContext(ApplicationContext ctx) throws BeansException {
        applicationContext = ctx;
        //初始化即加载,之所以放到该方法里，是因为static静态块比该方法先执行，会导致applicationContext对象为null
        loadMap();
    }
}
