package com.af.v4.system.restful.service;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.restful.exception.FileNotFoundException;
import com.af.v4.system.restful.logic.ExpressionHelper;
import com.af.v4.system.restful.mapper.ResourceMapper;
import com.af.v4.system.restful.mapper.SqlMapper;
import com.af.v4.system.restful.sql.SqlAction;
import com.af.v4.system.restful.utils.enums.ResourceType;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Iterator;
import java.util.Map;

@Service
@Transactional(rollbackFor = Exception.class)
public class SqlService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlService.class);

    private final ResourceMapper resourceMapper;

    private final SqlMapper sqlMapper;

    private final SqlAction sqlAction;

    public SqlService(ResourceMapper resourceMapper, SqlMapper sqlMapper, SqlAction sqlAction) {
        this.resourceMapper = resourceMapper;
        this.sqlMapper = sqlMapper;
        this.sqlAction = sqlAction;
    }

    /**
     * 获取SQL文件的编译后SQL
     *
     * @param name   SQL文件映射名
     * @param params 参数
     * @return SQL语句
     */
    public String getSQL(String name, JSONObject params) {
        //转为Map
        Map<String, Object> map = params.toMap();
        return this.call(name, map);
    }


    /**
     * 获取SQL语句的合计执行结果
     *
     * @param name : sql语句名
     * @param str  : sql语句执行参数
     * @return JSON格式Sql语句执行结果
     */
    public JSONObject queryTotal(String name, String str) {
        // 获取参数，求和字段等内容
        JSONObject param;
        Object sums = null;
        if (str != null && !str.isEmpty()) {
            JSONObject json = new JSONObject(str);
            if (json.has("data")) {
                param = json.getJSONObject("data");
            } else {
                param = new JSONObject();
            }
            if (json.has("sums")) {
                sums = json.get("sums");
            }
        } else {
            param = new JSONObject();
        }
        Map<String, Object> params = param.toMap();
        // 产生sql语句编译后的结果
        String sql = this.call(name, params);

        // 求和时，order by会导致sql错误，过滤掉order by部分。
        sql = filterOutOrderBy(sql, sums);
        JSONArray array = sqlAction.query(sql);
        return array.getJSONObject(0);
    }

    @Deprecated
    public JSONArray query(String sql) {
        return querySQL(sql);
    }

    /**
     * 执行sql分页查询
     */
    public JSONArray query(String name, String str) {
        return query(name, 0, 1000, str);
    }

    /**
     * 执行sql分页查询，参数为json对象
     */
    public JSONArray query(String name, JSONObject params) {
        Map<String, Object> map = params.toMap();
        return query(name, 1, 1000, map);
    }

    public JSONArray query(String name, JSONObject params, Integer pageSize) {
        Map<String, Object> map = params.toMap();
        return pageSize <= 0 ? query(name, 1, 9999999, map) : query(name, 1, pageSize, map);
    }

    /**
     * 执行sql分页查询，参数为map
     */
    public JSONArray query(String name, Map<String, Object> params) {
        return query(name, 1, 1000, params);
    }

    public JSONArray query(String name, Map<String, Object> params, Integer pageSize) {
        return pageSize <= 0 ? query(name, 1, 9999999, params) : query(name, 1, pageSize, params);
    }

    public JSONArray query(String name, int pageNo, int pageSize, String str) {
        // pageNo小于0， 纠正成1
        if (pageNo <= 0) {
            pageNo = 1;
        }

        // pageSize小于0，纠正成1, 解除1000限制
        if (pageSize < 1) {
            pageSize = 1000;
        }

        // 拿到json对象参数
        JSONObject param;
        if (str != null && !str.isEmpty()) {
            param = new JSONObject(str);
            param = param.getJSONObject("data");
        } else {
            param = new JSONObject();
        }
        Map<String, Object> params = param.toMap();
        return query(name, pageNo, pageSize, params);
    }

    public JSONArray query(String name, int pageNo, int pageSize,
                           Map<String, Object> params) {
        // 产生SQL语句编译后的结果
        String sql = this.call(name, params);

        JSONArray array = sqlAction.query(sql, pageNo - 1, pageSize);
        LOGGER.info("SQL[" + name + "] Result: \n" + array.toString());
        return array;
    }

    public JSONArray query(String name, String str, Integer pageSize) {
        return pageSize <= 0 ? query(name, 1, 9999999, str) : query(name, 1, pageSize, str);
    }

    public JSONArray querySQL(String sql) {
        return querySQL(sql, 1, 9999999, false);
    }

    public JSONArray querySQL(String sql, boolean isNoLog) {
        return querySQL(sql, 1, 9999999, isNoLog);
    }

    public JSONArray querySQL(String sql, Integer pageSize) {
        return querySQL(sql, pageSize, false);
    }

    public JSONArray querySQL(String sql, Integer pageSize, boolean isNoLog) {
        return pageSize <= 0 ? querySQL(sql, 1, 9999999, isNoLog) : querySQL(sql, 1, pageSize, isNoLog);
    }

    public JSONArray querySQL(String sql, int pageNo, int pageSize) {
        return querySQL(sql, pageNo, pageSize, false);
    }

    public JSONArray querySQL(String sql, int pageNo, int pageSize, boolean isNoLog) {
        JSONArray array = sqlAction.query(sql, pageNo - 1, pageSize, isNoLog);
        if (!isNoLog) {
            LOGGER.info("SQL Result: \n" + array.toString());
        }
        return array;
    }

    /**
     * 执行sql语句
     */
    public void run(String sql) throws NullPointerException {
        sqlAction.bulkSQLUpdate(sql);
    }

    public void run(String sql, Object... params) {
        String useSql = String.format(sql.replace("{}", "%s"), params);
        sqlAction.bulkSQLUpdate(useSql);
    }

    /**
     * 执行sql语句，非hql 返回受影响行数
     */
    public int runSQL(String sql) {
        return sqlAction.bulkSQLUpdate(sql);
    }

    public int runSQL(String sql, Object... params) {
        String useSql = String.format(sql.replace("{}", "%s"), params);
        return sqlAction.bulkSQLUpdate(useSql);
    }

    /**
     * 基于JSON对象的call函数
     */
    public String call(String sqlName, JSONObject json) {
        Map<String, Object> param = json.toMap();
        return call(sqlName, param);
    }

    /**
     * 调用其它sql语句, 产生sql语句经过参数处理的结果串
     */
    public String call(String sqlName, Map<String, Object> params) {
        // 获取原始sql语句
        String path = sqlMapper.getValue(sqlName);
        if (path == null) {
            throw new ServiceException("sql语句未注册！" + sqlName);
        }
        String sql;
        try {
            sql = resourceMapper.getString(ResourceType.SQL, sqlName, path);
            //如果使用了新格式化参数写法，需要替换为解析语法树支持的写法
            if (sql.contains("${")) {
                sql = sql.replace("${", "{");
            }
        } catch (FileNotFoundException e) {
            throw new ServiceException(path + ".文件无配置");
        }
        // 把自身注册到执行环境中
        params.put("sql", this);
        sql = ExpressionHelper.run(sql, params).toString();
        return sql;
    }

    /**
     * 过滤order by子句，产生求和结果
     */
    private String filterOutOrderBy(String source, Object sums) {
        int idx = source.toLowerCase().lastIndexOf("order by");
        if (idx == -1) {
            idx = source.length() - 1;
        }
        StringBuilder sql = new StringBuilder("select ");
        // 如果有求和部分，产生求和部分的语句
        if (sums != null) {
            if (sums instanceof JSONArray arraySums) {
                for (int i = 0; i < arraySums.length(); i++) {
                    String name = (String) arraySums.get(i);
                    sql.append("sum(").append(name).append(") ").append(name).append(", ");
                }
            } else if (sums instanceof JSONObject objSums) {
                Iterator<String> keys = objSums.keys();
                while (keys.hasNext()) {
                    String name = keys.next();
                    String value = objSums.getString(name);
                    sql.append("sum(").append(value).append(") ").append(name).append(", ");
                }
            }
        }
        if (idx != -1) {
            sql.append("count(*) n, 1 placeholder from ( ").append(source, 0, idx).append(") t_");
        }
        return sql.toString();
    }
}
