package com.af.v4.system.restful.service;

import com.af.v4.system.api.RemoteLogicService;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.restful.exception.FileNotFoundException;
import com.af.v4.system.restful.logic.ExpressionHelper;
import com.af.v4.system.restful.mapper.LogicMapper;
import com.af.v4.system.restful.mapper.PluginMapper;
import com.af.v4.system.restful.mapper.ResourceMapper;
import com.af.v4.system.restful.transaction.SessionPool;
import com.af.v4.system.restful.utils.ExceptionHelper;
import com.af.v4.system.restful.utils.enums.ResourceType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.stream.Collectors;

@Service
public class LogicService {
    private static final Logger LOGGER = LogManager.getLogger(LogicService.class);
    private final SqlService sqlService;
    private final SessionPool sessionPool;
    private final ResourceMapper resourceMapper;
    private final RedisService redisService;
    private final RemoteLogicService remoteLogicService;
    private final LocalEntityService entityService;
    private final PluginMapper pluginMapper;
    private final LogicMapper logicMapper;

    public LogicService(SqlService sqlService, SessionPool sessionPool, ResourceMapper resourceMapper, RedisService redisService, RemoteLogicService remoteLogicService, LocalEntityService entityService, PluginMapper pluginMapper, LogicMapper logicMapper) {
        this.sqlService = sqlService;
        this.sessionPool = sessionPool;
        this.resourceMapper = resourceMapper;
        this.redisService = redisService;
        this.remoteLogicService = remoteLogicService;
        this.entityService = entityService;
        this.pluginMapper = pluginMapper;
        this.logicMapper = logicMapper;
    }

    /**
     * 执行业务逻辑处理过程
     */
    @Transactional(rollbackFor = Exception.class)
    public Object run(String name, String str) {
        JSONObject param = null;
        // 把传递过来的参数，放到data里，以便跟entity，sql等对象区别开来
        // 判断传来的数据是否是XML格式
        str = str.trim();
        if (str.startsWith("<") && str.endsWith(">")) {
            // 传来数据为XML格式
            param = new JSONObject();
            param.put("xml", str);
        } else {
            // 传来数据为JSON格式
            Object json = new JSONTokener(str).nextValue();
            if (json instanceof JSONObject st) {
                param = st;
                if (!param.isNull("data")) {
                    Object dataParam = param.get("data");
                    if (dataParam instanceof JSONObject) {
                        param = (JSONObject) dataParam;
                        param.put("standardData", new JSONObject(str).toString());
                    }
                }
            } else if (json instanceof JSONArray ar) {
                param = new JSONObject();
                param.put("arr", ar);
            }
        }

        return run(name, param);
    }

    @Transactional(rollbackFor = Exception.class)
    public Object run(String name, JSONObject param) {
        String message = "维护信息:开始执行Logic[" + name + "]，参数：" + param;
        LOGGER.debug(message);
        // 获取源程序内容
        String path = logicMapper.getValue(name);
        if (path == null) {
            throw new ServiceException("业务逻辑未找到: " + name);
        }

        String source;
        try {
            source = resourceMapper.getString(ResourceType.LOGIC, name, path);
        } catch (FileNotFoundException e) {
            throw new ServiceException(path + ".文件无配置");
        }
        // 附加用户注册的对象到业务逻辑中
        Map<String, Object> params = pluginMapper.getAllMap().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, object -> object.getValue().get("value")));
        // 执行源程序
        params.put("data", param);
        // 附加entityServer, sqlServer等对象到参数中
        params.put("log", LOGGER);
        params.put("entity", entityService);
        params.put("sql", sqlService);
        params.put("session", sessionPool.getSession());
        params.put("redis", redisService);
        // 把LogicServer自身注册进去
        params.put("logic", this);
        // 执行结果
        Object result;
        long begin = System.currentTimeMillis();
        try {
            result = ExpressionHelper.run(source, params);
            return result;
        } finally {
            long end = System.currentTimeMillis();
            long time = end - begin;
            String engMessage = "维护信息:执行Logic[" + name + "]耗时:" + time;
            if (time >= 8000) {
                LOGGER.error(engMessage);
            } else if (time >= 4000) {
                LOGGER.warn(engMessage);
            } else {
                LOGGER.debug(engMessage);
            }
        }
    }

    /**
     * web层运行业务逻辑入口
     *
     * @param logicName 业务逻辑名
     * @param values    业务逻辑参数
     * @return 运行结果
     */
    @Transactional(rollbackFor = Exception.class)
    public Object runMain(String logicName, String values) {
        try {
            Object result = this.run(logicName, values);
            if (result == null) {
                return "";
            }
            return result;
        } catch (Exception ex) {
            String stack = ExceptionLogService.getErrorInfo(ExceptionHelper.stackToString(ex), 1);
            ServiceException serviceException = null;
            if (ex instanceof ServiceException e) {
                serviceException = e;
            } else {
                if (ex.getCause() instanceof InvocationTargetException targetException) {
                    if (targetException.getTargetException() instanceof ServiceException e) {
                        serviceException = e;
                    }
                }
            }
            if (serviceException != null) {
                serviceException.setDetailMessage(stack);
            } else {
                serviceException = new ServiceException();
                serviceException.setMessage("服务器内部错误");
                serviceException.setDetailMessage(stack);
            }
            throw serviceException;
        }
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, String params) {
        R<Object> logicResult = remoteLogicService.run(serviceName, logicName, params, SecurityConstants.INNER);
        return logicResult.parseResponseJson();
    }

    /**
     * 远程调用Logic
     *
     * @param serviceName 服务名
     * @param logicName   Logic名称
     * @param params      Logic JSON参数
     * @return 执行结果
     */
    //    @GlobalTransactional
    public JSONObject remoteRun(String serviceName, String logicName, JSONObject params) {
        return remoteRun(serviceName, logicName, params.toString());
    }
}
