package com.af.v4.system.restful.mapper;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.plugins.io.IOTools;
import com.af.v4.system.restful.exception.FileNotFoundException;
import com.af.v4.system.restful.utils.enums.ResourceType;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentHashMap;

/**
 * 资源映射器
 *
 * @author Mr.river
 */
@Component
public record ResourceMapper(RedisService redisService) {

    /**
     * Logic源缓存
     */
    private static final ConcurrentHashMap<String, String> CACHE_LOGIC_MAP;
    /**
     * SQL源缓存
     */
    private static final ConcurrentHashMap<String, String> CACHE_SQL_MAP;
    public static CACHE_MODE mode = CACHE_MODE.NORMAL;

    static {
        CACHE_LOGIC_MAP = new ConcurrentHashMap<>(10);
        CACHE_SQL_MAP = new ConcurrentHashMap<>(10);
    }

    /**
     * 获取资源文件里，指定全路径文件的文本内容
     *
     * @param fileName : 以'/'开始的资源文件的全路径文件名
     * @return 文件的文本内容
     */
    public static String getString(String fileName) throws FileNotFoundException {
        return IOTools.getStringByReader(fileName, bufferedReader -> {
            String line;
            StringBuilder sb = new StringBuilder();
            while ((line = bufferedReader.readLine()) != null) {
                sb.append(line).append("\r\n");
            }
            return sb;
        }, errorPath -> {
            throw new FileNotFoundException(fileName + "文件不存在", fileName);
        }).toString();
    }

    /**
     * Logic缓存清理
     */
    public static void clearLogicCache() {
        CACHE_LOGIC_MAP.clear();
    }

    /**
     * SQL缓存清理
     */
    public static void clearSqlCache() {
        CACHE_SQL_MAP.clear();
    }

    /**
     * 获取资源文件里，指定全路径文件的文本内容，并缓存
     *
     * @param fileName : 以'/'开始的资源文件的全路径文件名
     * @return 文件的文本内容
     */
    public String getString(ResourceType resourceType, String alias, String fileName) throws FileNotFoundException {
        String source;
        switch (resourceType) {
            case LOGIC:
                if (mode == CACHE_MODE.NORMAL && CACHE_LOGIC_MAP.containsKey(alias)) {
                    return CACHE_LOGIC_MAP.get(alias);
                } else {
                    String sourceStr = getString(fileName);
                    // 处理回车换行
                    sourceStr = sourceStr.replace("\r\n", "\n");
                    if (mode == CACHE_MODE.NORMAL) {
                        CACHE_LOGIC_MAP.put(alias, sourceStr);
                    }
                    return sourceStr;
                }
            case SQL:
                if (mode == CACHE_MODE.NORMAL && CACHE_SQL_MAP.containsKey(alias)) {
                    return CACHE_SQL_MAP.get(alias);
                } else {
                    source = getString(fileName);
                    // 获取编译后的sql语句
                    source = source.replace("\r\n", "\n");
                    source = "$" + source;
                    if (mode == CACHE_MODE.NORMAL) {
                        CACHE_SQL_MAP.put(alias, source);
                    }
                    return source;
                }
            default:
                throw new ServiceException("错误的缓存数据类型");
        }
    }

    public enum CACHE_MODE {
        //正常模式
        NORMAL,
        //DEBUG模式，不开启缓存
        DEBUG
    }
}
