package com.af.v4.system.plugins.http;

import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.protocol.HTTP;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * HttpClient连接池实现
 *
 * @author Mr.river
 * @see RestTools
 */
public class HttpConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger logger = LoggerFactory.getLogger(HttpConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    /**
     * 发送HTTP请求的客户端单例
     */
    private static volatile CloseableHttpClient httpClient;

    /**
     * 获取httpclient的https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpClient createHttpClient(LayeredConnectionSocketFactory sslSocketFactory) {
        //设置https相关信息
        if (sslSocketFactory == null) {
            logger.debug("创建http客户端会话");
            sslSocketFactory = SSLConnectionSocketFactory.getSocketFactory();
        } else {
            logger.debug("创建https客户端会话");
        }
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", sslSocketFactory)
                .build();

        /* 设置连接参数 **/
        PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager(registry);
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);

        //设置keepalive
        ConnectionKeepAliveStrategy myStrategy = (response, context) -> {
            // Honor 'keep-alive' header
            HeaderElementIterator it = new BasicHeaderElementIterator(
                    response.headerIterator(HTTP.CONN_KEEP_ALIVE));
            while (it.hasNext()) {
                HeaderElement he = it.nextElement();
                String param = he.getName();
                String value = he.getValue();
                if (value != null && "timeout".equalsIgnoreCase(param)) {
                    try {
                        return Long.parseLong(value) * 1000;
                    } catch (NumberFormatException ignore) {
                    }
                }
            }
            // Keep alive for 5 seconds only
            return 5 * 1000;
        };

        return HttpClients.custom()
                .setKeepAliveStrategy(myStrategy)
                .setConnectionManagerShared(true)
                .setConnectionManager(manager)
                .setRetryHandler(new DefaultHttpRequestRetryHandler(0, false))
                .evictExpiredConnections()
                .evictIdleConnections(HttpClientConfig.HTTP_IDEL_TIME_OUT, TimeUnit.MILLISECONDS)
                .build();
    }

    /**
     * 获取HTTP请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (SYNC_LOCK) {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            }
        }
        return httpClient;
    }

    /**
     * 发送通用HTTP请求
     *
     * @param path       请求路径
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     * @return 请求结果
     */
    public static String request(String path, String value, String headersStr, HttpClientConfig config, HttpRequestBase base) {
        return request(path, value, headersStr, config, base, null);
    }

    public static String request(String path, String value, String headersStr, HttpRequestBase base) {
        return request(path, value, headersStr, null, base, null);
    }

    /**
     * 发送通用HTTP请求
     *
     * @param path       请求路径
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     * @param httpClient client对象
     * @return 请求结果
     */
    public static String request(String path,
                                 String value,
                                 String headersStr,
                                 HttpClientConfig config,
                                 HttpRequestBase base,
                                 CloseableHttpClient httpClient) {
        logger.info("请求地址：" + path);
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base instanceof HttpEntityEnclosingRequestBase entityEnclosingRequestBase) {
            if (entityEnclosingRequestBase.getEntity() == null) {
                setBody(entityEnclosingRequestBase, value);
            }
        }
        String result;
        long begin = System.currentTimeMillis();
        try (CloseableHttpResponse response = httpClient.execute(base, HttpClientContext.create())) {
            result = getResponseData(begin, response);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result;
    }

    public static String request(String path,
                                 String value,
                                 String headersStr,
                                 HttpRequestBase base,
                                 CloseableHttpClient httpClient) {
        return request(path, value, headersStr, null, base, httpClient);
    }

    public static String requestFormUrlEncoded(String path,
                                               JSONObject params,
                                               HttpEntityEnclosingRequestBase base,
                                               CloseableHttpClient client) throws IOException {
        setUrlEncodedBody(base, params);
        return request(path, params.toString(), null, null, base, client);
    }

    public static String requestFormUrlEncoded(String path,
                                               Map<String, String> params,
                                               HttpEntityEnclosingRequestBase base,
                                               CloseableHttpClient client) throws IOException {
        return requestFormUrlEncoded(path, new JSONObject(params), base, client);
    }
}
