package com.af.v4.system.plugins.http;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.nio.reactor.ConnectingIOReactor;
import org.apache.http.nio.reactor.IOReactorException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;
import java.util.Map;

/**
 * HttpAsyncClient连接池实现
 *
 * @author Mr.river
 * @see RestAsyncTools
 */
public class HttpAsyncConnectionPoolUtil extends BaseHttpPoolUtil {

    private static final Logger logger = LoggerFactory.getLogger(HttpAsyncConnectionPoolUtil.class);
    /**
     * 相当于线程锁,用于线程安全
     */
    private final static Object SYNC_LOCK = new Object();
    /**
     * 连接池工具类单例
     */
    private static volatile BaseHttpPoolUtil httpConnectionPoolUtil;
    /**
     * 发送HTTP异步请求的客户端单例
     */
    private static volatile CloseableHttpAsyncClient httpClient;

    /**
     * 获取httpclient的异步https实例（需要SSL证书）
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient(SSLIOSessionStrategy sslSocketFactory) {
        return createHttpClient(sslSocketFactory);
    }

    /**
     * 构建httpclient实例
     *
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient() {
        return createHttpClient(null);
    }

    /**
     * 构建httpclient实例
     *
     * @param sslSocketFactory SSL构建器
     * @return httpclient实例
     */
    private static CloseableHttpAsyncClient createHttpClient(SSLIOSessionStrategy sslSocketFactory) {
        //设置https相关信息
        if (sslSocketFactory == null) {
            logger.debug("创建HTTP异步客户端会话");
            sslSocketFactory = SSLIOSessionStrategy.getDefaultStrategy();
        } else {
            logger.debug("创建https异步客户端会话");
        }
        Registry<SchemeIOSessionStrategy> registry = RegistryBuilder.<SchemeIOSessionStrategy>create()
                .register("http", NoopIOSessionStrategy.INSTANCE)
                .register("https", sslSocketFactory)
                .build();
        // 配置io线程
        IOReactorConfig ioReactorConfig = IOReactorConfig.custom().
                setIoThreadCount(Runtime.getRuntime().availableProcessors() * 2)
                .setSoKeepAlive(true)
                .build();
        // 设置连接池大小
        ConnectingIOReactor ioReactor;
        try {
            ioReactor = new DefaultConnectingIOReactor(ioReactorConfig);
        } catch (IOReactorException e) {
            throw new RuntimeException(e);
        }
        PoolingNHttpClientConnectionManager manager = new PoolingNHttpClientConnectionManager(ioReactor, registry);

        /* 设置连接参数 **/
        // 最大连接数
        manager.setMaxTotal(HttpClientConfig.HTTP_MAX_POOL_SIZE);
        // 路由最大连接数
        manager.setDefaultMaxPerRoute(HttpClientConfig.HTTP_MAX_POOL_SIZE);

        return HttpAsyncClients.custom()
                .setConnectionManager(manager)
                .build();
    }

    /**
     * 获取HTTP异步请求 httpclient实例
     *
     * @return httpclient实例
     */
    public static CloseableHttpAsyncClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (SYNC_LOCK) {
                if (httpClient == null) {
                    httpClient = createHttpClient();
                }
            }
        }
        httpClient.start();
        return httpClient;
    }

    /**
     * 发送通用HTTP异步请求
     *
     * @param path       请求路径
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     */
    public static void request(String path, String value, String headersStr, HttpClientConfig config, HttpRequestBase base) {
        request(path, value, headersStr, config, base, null, null);
    }

    public static void request(String path, String value, String headersStr, HttpRequestBase base) {
        request(path, value, headersStr, null, base, null, null);
    }

    public static void request(String path, String value, String headersStr, HttpRequestBase base, FutureCallback<HttpResponse> futureCallback) {
        request(path, value, headersStr, null, base, null, futureCallback);
    }

    /**
     * 发送通用HTTP异步请求
     *
     * @param path       请求路径
     * @param value      请求参数
     * @param headersStr 请求头
     * @param base       请求类型
     * @param httpClient client对象
     */
    public static void request(String path,
                               String value,
                               String headersStr,
                               HttpClientConfig config,
                               HttpRequestBase base,
                               CloseableHttpAsyncClient httpClient,
                               FutureCallback<HttpResponse> futureCallback) {
        logger.info("异步请求地址：" + path);
        if (httpClient == null) {
            httpClient = getHttpClient();
        }
        if (futureCallback == null) {
            futureCallback = new InnerAsyncResponse();
        }
        //配置请求参数
        setRequestConfig(config, base);
        //设置请求地址
        base.setURI(URI.create(path));
        //设置请求头
        setHeaders(base, headersStr);
        //设置请求体
        if (base instanceof HttpEntityEnclosingRequestBase entityEnclosingRequestBase) {
            if (entityEnclosingRequestBase.getEntity() == null) {
                setBody(entityEnclosingRequestBase, value);
            }
        }
        httpClient.execute(base, futureCallback);
    }

    public void requestFormUrlEncoded(String path,
                                      JSONObject params,
                                      HttpEntityEnclosingRequestBase base,
                                      CloseableHttpAsyncClient client,
                                      FutureCallback<HttpResponse> futureCallback) throws IOException {
        setUrlEncodedBody(base, params);
        request(path, params.toString(), null, null, base, client, futureCallback);
    }

    public void requestFormUrlEncoded(String path,
                                      JSONObject params,
                                      HttpEntityEnclosingRequestBase base,
                                      CloseableHttpAsyncClient client) throws IOException {
        requestFormUrlEncoded(path, params, base, client, null);
    }

    public void requestFormUrlEncoded(String path,
                                      Map<String, String> params,
                                      HttpEntityEnclosingRequestBase base,
                                      CloseableHttpAsyncClient client,
                                      FutureCallback<HttpResponse> futureCallback) throws IOException {
        requestFormUrlEncoded(path, new JSONObject(params), base, client, futureCallback);
    }

    public void requestFormUrlEncoded(String path,
                                      Map<String, String> params,
                                      HttpEntityEnclosingRequestBase base,
                                      CloseableHttpAsyncClient client) throws IOException {
        requestFormUrlEncoded(path, params, base, client, null);
    }

    public static class InnerAsyncResponse implements FutureCallback<HttpResponse> {

        final long begin = System.currentTimeMillis();

        @Override
        public void completed(HttpResponse response) {
            String result = null;
            try {
                result = getResponseData(begin, response);
            } catch (Exception e) {
                this.failed(e);
            }
            logger.info("异步请求结果：" + result);
        }

        @Override
        public void failed(Exception ex) {
            logger.error("异步请求时发生异常：", ex);
        }

        @Override
        public void cancelled() {

        }
    }
}
