package com.af.v4.system.plugins.date;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;

import java.util.Calendar;
import java.util.Date;

/**
 * 日期工具类
 *
 * @author Mr.river
 * @see DateUtil hutool日期工具类实现
 */
public class DateTools {

    /**
     * 获取当前时间
     *
     * @return 当前时间  格式:标准格林威治时间（GMT）
     */
    public static Date getNow() {
        return DateUtil.date();
    }

    /**
     * 获取当前时间
     *
     * @return 当前时间  格式:yyyy-MM-dd HH:mm:ss
     */
    public static String getNow2() {
        return DateUtil.now();
    }

    /**
     * 获取当前时间
     *
     * @return 当前时间  返回值long类型
     */
    public static long getCurrentTimeMillis() {
        return System.currentTimeMillis();
    }

    /**
     * 根据指定格式获取当前时间
     *
     * @param format format字符串
     * @return 当前时间  返回指定格式的String字符串
     */
    public static String getNow(String format) {
        return DateUtil.date().toString(format);
    }

    /**
     * 返回当前时间的年份
     *
     * @return 当前时间的年份
     */
    public static String getNowYear() {
        return String.valueOf(DateUtil.thisYear());
    }

    /**
     * 返回当前时间的月份
     *
     * @return 当前时间的月份
     */
    public static String getNowMonth() {
        return String.valueOf(getNowMonthAsInt());
    }

    /**
     * 返回当前时间的月份
     *
     * @return 当前时间的月份
     */
    public static int getNowMonthAsInt() {
        return DateUtil.thisMonth() + 1;
    }

    /**
     * 返回当前时间的日
     *
     * @return 当前时间的日
     */
    public static String getNowDay() {
        return String.valueOf(DateUtil.thisDayOfMonth());
    }

    /**
     * 返回当前时间是星期几
     *
     * @return 星期几 , 1是星期一
     */
    public static String getNowDayOfWeek() {
        int value = DateUtil.thisDayOfWeek() - 1;
        if (value == 0) {
            value = 7;
        }
        return String.valueOf(value);
    }

    /**
     * 获取指定时间的月份
     *
     * @param date 指定时间字符串
     * @return 指定时间的月份
     */
    public static int getMonth(String date) {
        return DateUtil.parse(date).month() + 1;
    }

    /**
     * 获取指定时间的日份
     *
     * @param date 指定时间字符串
     * @return 指定时间的日份
     */
    public static int getDate(String date) {
        return DateUtil.parse(date).dayOfMonth();
    }

    /**
     * 返回传入时间的年份和月份
     *
     * @param date 指定时间字符串
     * @return 传入的年份和月份，格式为yyyyMM
     */
    public static String getYearMonth(String date) {
        return DateUtil.parse(date).toString(DatePattern.SIMPLE_MONTH_FORMAT);
    }

    /**
     * 格式化一个日期为指定日期格式
     *
     * @param datetime 日期字符串
     * @return 标准日期格式 返回值为指定的String格式
     */
    public static String format(String datetime, String format) {
        return DateUtil.parse(datetime).toString(format);
    }

    /**
     * 格式化一个日期为标准日期格式
     *
     * @param datetime 日期字符串
     * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
     */
    public static String formatDateTime(String datetime) {
        return format(datetime, DatePattern.PURE_DATETIME_PATTERN);
    }

    /**
     * 格式化一个指定格式的日期字符串为标准日期格式
     *
     * @param dateStr 日期字符串
     * @param format  该字符串的格式
     * @return 标准日期格式  返回值为指定的String格式::yyyy-MM-dd HH:mm:ss
     */
    public static String formatDateTime(String dateStr, String format) {
        return DateUtil.parse(dateStr, format).toString();
    }

    /**
     * 格式化一个指定格式的日期字符串为指定日期格式
     *
     * @param dateStr   日期字符串
     * @param format    该字符串的格式
     * @param newFormat 指定转换的日期格式
     */
    public static String formatDateTime(String dateStr, String format, String newFormat) {
        return DateUtil.parse(dateStr, format).toString(newFormat);
    }

    /**
     * 比较两个日期的大小
     *
     * @param dateStr1 第一个日期字符串
     * @param dateStr2 第二个日期字符串
     * @return a>=b时为true
     */
    public static boolean compareDate(String dateStr1, String dateStr2) {
        return DateUtil.compare(DateUtil.parse(dateStr1), DateUtil.parse(dateStr2)) >= 0;
    }

    /**
     * 比较两个日期(指定日期格式)的大小
     *
     * @param dateStr1  第一个日期字符串
     * @param dateStr2  第二个日期字符串
     * @param formatStr 指定比较日期的格式
     * @return dateStr1>=dateStr2时为true
     */
    public static boolean compareFormatDate(String dateStr1, String dateStr2, String formatStr) {
        return DateUtil.compare(DateUtil.parse(dateStr1, formatStr), DateUtil.parse(dateStr2, formatStr)) >= 0;
    }

    /**
     * 获取指定日期的开始时间
     *
     * @param date   指定日期
     * @param type   操作类型，可选值(year,month,date)
     * @param format 指定日期格式
     * @return 当前年/月/日的指定格式开始时间
     */
    public static String getStartDate(String date, String type, String format) {
        if (StrUtil.isBlank(date) || StrUtil.isBlank(type)) {
            throw new RuntimeException("获取开始时间: 您所传的参数有误，请检查参数是否正确！！");
        }
        DateTime dateTime = DateUtil.parse(date);
        if ("month".equalsIgnoreCase(type)) {
            return DateUtil.beginOfMonth(dateTime).toString(format);
        } else if ("year".equalsIgnoreCase(type)) {
            return DateUtil.beginOfYear(dateTime).toString(format);
        } else {
            return DateUtil.beginOfDay(dateTime).toString(format);
        }
    }

    public static String getStartDate(String date, String type) {
        return getStartDate(date, type, DatePattern.PURE_DATETIME_PATTERN);
    }

    /**
     * 获取指定日期的最后时间
     *
     * @param date   指定日期
     * @param type   操作类型，可选值(year,month,date)
     * @param format 格式化日期
     * @return 当前年/月/日的指定格式最后时间
     */
    public static String getEndDate(String date, String type, String format) {
        if (StrUtil.isBlank(date) || StrUtil.isBlank(type)) {
            throw new RuntimeException("获取最后时间: 您所传的参数有误，请检查参数是否正确！！");
        }
        DateTime dateTime = DateUtil.parse(date);
        if ("month".equalsIgnoreCase(type)) {
            return DateUtil.endOfMonth(dateTime).toString(format);
        } else if ("year".equalsIgnoreCase(type)) {
            return DateUtil.endOfYear(dateTime).toString(format);
        } else {
            return DateUtil.endOfDay(dateTime).toString(format);
        }
    }

    public static String getEndDate(String date, String type) {
        return getEndDate(date, type, DatePattern.PURE_DATETIME_PATTERN);
    }

    /**
     * 获取阶梯开始日期
     *
     * @param start    开始时间 格式为MM-dd 或 yyyy-HH-mm HH:mm:ss
     * @param cycle    阶梯周期
     * @param handDate 下发时间 格式为yyyy-MM-dd
     * @return 返回阶梯开始日期 yyyy-MM-dd
     * @throws Exception 异常
     */
    public static String getStairStartDate(String start, Object cycle, String handDate) throws Exception {
        // 参数验证
        if ("".equals(start) || start == null) {
            throw new Exception("阶梯开始时间不能为空!!");
        }
        if (cycle == null) {
            throw new Exception("阶梯周期不能为空!!");
        }
        if ("".equals(handDate) || handDate == null) {
            throw new Exception("抄表时间不能为空!!");
        }

        Calendar now = Calendar.getInstance();
        if (start.length() > 5) {
            //如果长度大于5，则按传入yyyy-HH-mm HH:mm:ss格式进行截取
            start = start.substring(5, 10);
        }
        String[] str = start.split("-");
        // 开始月份
        int beginMonth = Integer.parseInt(str[0]);
        // 当前月份
        int nowMonth = getMonth(handDate);

        if (isLastYear(handDate)) {
            now.set(Calendar.YEAR, now.get(Calendar.YEAR) - 1);
        } else {
            now.set(Calendar.YEAR, now.get(Calendar.YEAR));
        }

        now.set(Calendar.DAY_OF_MONTH, Integer.parseInt(str[1]));

        // 计算月份
        int a = (nowMonth + (12 - beginMonth)) % (Integer.parseInt(cycle.toString()));

        int month;
        boolean isLowerDay = Calendar.getInstance().get(Calendar.DATE) - Integer.parseInt(str[1]) < 0;
        if (isLowerDay) {
            month = a == 0 ? a - (int) cycle : -a;
        } else {
            month = -a;
        }

        now.add(Calendar.MONTH, month);

        return DateUtil.format(now.getTime(), DatePattern.NORM_DATE_FORMAT);
    }

    /**
     * 比较指定日期和当前日期的年份
     *
     * @param date 指定日期，格式为yyyy-MM-dd
     * @return 比较结果，true：指定的日期年份比当前日期年份早
     */
    private static boolean isLastYear(String date) {
        return DateUtil.parse(date).year() < DateUtil.date().year();
    }

    /**
     * 根据操作类型获取当前日期的前后日期，如上周或下个月的这一天
     *
     * @param type   操作类型，可选值(YEAR,MONTH,DATE,HOUR,MINUTE,SECOND)
     * @param param  差值，负数即从当前日期往前推
     * @param format 返回的日期格式，默认为yyyy-MM-dd
     * @return 当前日期的前后日期
     */
    public static String getDiffDate(String type, String param, String format) {
        DateTime dateTime = DateUtil.date();
        return getDelayDate(dateTime, type, param, format);
    }

    /**
     * 根据操作类型获取指定日期的前后日期，如上周或下个月的这一天
     *
     * @param date  指定日期，格式为标准日期格式
     * @param type  操作类型，可选值(YEAR,MONTH,DAY)
     * @param param 差值，负数即从当前日期往前推
     * @return 当前日期的前后日期，格式为标准日期格式
     */
    public static String getDelayDate(String date, String type, String param) {
        DateTime dateTime = DateUtil.parse(date);
        return getDelayDate(dateTime, type, param, DatePattern.PURE_DATETIME_PATTERN);
    }

    /**
     * 获取两个时间的相差间隔
     *
     * @param date1 时间1
     * @param date2 时间2
     * @return 相差间隔（毫秒，绝对值）
     */
    public static String getDateBetween(String date1, String date2) {
        return String.valueOf(DateUtil.betweenMs(DateUtil.parse(date1), DateUtil.parse(date2)));
    }

    /**
     * 获取两个时间的天数相差间隔
     *
     * @param date1   时间1
     * @param date2   时间2
     * @param isReset * 有时候我们计算相差天数的时候需要忽略时分秒。
     *                * 比如：2016-02-01 23:59:59和2016-02-02 00:00:00相差一秒
     *                * 如果isReset为<code>false</code>相差天数为0。
     *                * 如果isReset为<code>true</code>相差天数将被计算为1
     * @return 相差间隔（天数，绝对值）
     */
    public static String getDateDayBetween(String date1, String date2, Boolean isReset) {
        return String.valueOf(DateUtil.betweenDay(DateUtil.parse(date1), DateUtil.parse(date2), isReset));
    }

    private static String getDelayDate(DateTime dateTime, String type, String param, String format) {
        //获取处理后的时间
        switch (type.toUpperCase()) {
            case "YEAR" -> dateTime.offset(DateField.YEAR, Integer.parseInt(param));
            case "MONTH" -> dateTime.offset(DateField.MONTH, Integer.parseInt(param));
            case "DATE" -> dateTime.offset(DateField.DAY_OF_MONTH, Integer.parseInt(param));
            case "HOUR" -> dateTime.offset(DateField.HOUR, Integer.parseInt(param));
            case "MINUTE" -> dateTime.offset(DateField.MINUTE, Integer.parseInt(param));
            case "SECOND" -> dateTime.offset(DateField.SECOND, Integer.parseInt(param));
            default -> throw new RuntimeException("不支持的日期类型！可选值(YEAR,MONTH,DATE,HOUR,MINUTE,SECOND)");
        }
        return dateTime.toString(format);
    }
}
