package com.af.v4.system.common.task.product;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.task.consumer.TaskManageHandler;
import com.af.v4.system.common.task.entity.Task;
import com.af.v4.system.common.task.entity.TaskHandler;
import com.af.v4.system.common.task.entity.vo.TaskResult;
import com.af.v4.system.common.task.enums.TaskStatus;
import com.af.v4.system.common.task.event.TaskEvent;
import com.af.v4.system.common.task.service.TaskService;
import com.lmax.disruptor.*;
import com.lmax.disruptor.dsl.ProducerType;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.*;

/**
 * 事件 生产者
 */
@Component
public class TaskProduct {

    private final static Logger LOGGER = LoggerFactory.getLogger(TaskProduct.class);

    private static RingBuffer<TaskEvent> ringBuffer;
    // 任务id 与 ringBuffer 序列中的 消息载体绑定
    public static Map<Integer, Long> taskIdRinBufferSequence = new HashMap<>();

    public static Integer DEFAULT_BUFFER_SIZE = 1024 * 1024;
    public static JSONObject config = new JSONObject();

    @Autowired
    private ApplicationContext applicationContext;
    @Autowired
    private LiuLiConfigService liuLiConfigService;

    @Autowired
    private LogicService logicService;

    @PostConstruct
    public void init() {

        config = liuLiConfigService.get(TaskService.CONFIG_NAME);
        LOGGER.info(">>>> liuluConfig: " + config);
        DEFAULT_BUFFER_SIZE = config.optInt("ringBufferSize", 1024 * 1024);
        int corePoolSize = config.getInt("corePoolSize");
        int maximumPoolSize = config.getInt("maximumPoolSize");

        ExecutorService executor = new ThreadPoolExecutor(corePoolSize, maximumPoolSize, 0L, TimeUnit.MILLISECONDS,
                new ArrayBlockingQueue<>(10), new ThreadFactory() {
            private int counter = 0;

            @Override
            public Thread newThread(Runnable r) {
                String prefix = "TASK";
                return new Thread(r, prefix + "-" + counter++);
            }
        });
        // 设置多消费者, 广播模式 消费者实现WorkHandler 处理重复消费
        WorkHandler<TaskEvent>[] consumers = new TaskManageHandler[maximumPoolSize];
        for (int i = 0; i < consumers.length; i++) {
            TaskManageHandler taskManageHandler = applicationContext.getBean(TaskManageHandler.class);
            consumers[i] = taskManageHandler;
        }
        // 多生产者
        ringBuffer = RingBuffer.create(ProducerType.MULTI, TaskEvent::new,
                DEFAULT_BUFFER_SIZE, new YieldingWaitStrategy());
        SequenceBarrier barrier = ringBuffer.newBarrier();
        WorkerPool<TaskEvent> workerPool = new WorkerPool<>(ringBuffer, barrier, new IgnoreExceptionHandler(), consumers);
        ringBuffer.addGatingSequences(workerPool.getWorkerSequences());
        workerPool.start(executor);
    }

    /**
     * ringBuffer 写数据
     *
     * @param task        任务实体
     * @param taskHandler 任务实现
     */
    public TaskResult<Task> publishEvent(Task task, TaskHandler taskHandler) {

        // 获取下一个序列
        long sequence = ringBuffer.next();
        // 队列剩余空间
        LOGGER.info(">>>> ringBuffer队列当前序列: " + sequence % DEFAULT_BUFFER_SIZE);
        LOGGER.info(">>>> ringBuffer剩余容量: " + ringBuffer.remainingCapacity());
        // 获取事件
        TaskEvent event = ringBuffer.get(sequence);

        event.setTemplateTaskHandler(taskHandler);
        event.setTemplateTask(task);
        taskIdRinBufferSequence.put(task.getId(), sequence);

        // 发布事件
        ringBuffer.publish(sequence);
        return new TaskResult<Task>(TaskStatus.SUCCESS, task).setCode(200);
    }

    /**
     * 中断任务
     * 正在执行中的 任务将任务线程关闭, 未执行的任务移出 ringBuffer队列
     *
     * @param id task_id
     */
    public TaskResult<String> interrupt(Integer id) {

        TaskResult<String> taskResult = new TaskResult<>(TaskStatus.SUCCESS, "");
        // 任务 Sequence 序列编号
        Long sequence = taskIdRinBufferSequence.get(id);
        if (sequence == null) {
            taskResult.setMsg(TaskStatus.SUCCESS).setData("任务已执行完成");
            return taskResult;
        }
        TaskEvent taskEvent = ringBuffer.get(sequence);
        // 执行任务 本身 待中断的代码
        TaskResult<String> taskInterrupt = taskEvent.interruptTask(id);
        if (taskInterrupt.getMsg().getStatus().equals(TaskStatus.INTERRUPT.getStatus())) {
            // 任务状态为 `执行中`
            Thread thread = TaskManageHandler.threadMap.get(id);
            if (thread != null) {
                thread.interrupt();
            } else { // 任务状态为 `执行中-等待队列`
                ringBuffer.removeGatingSequence(new Sequence(sequence));
            }
            Timestamp timestamp = new Timestamp(System.currentTimeMillis());
            Task task = new Task();
            task.setId(id);
            task.setEndTime(timestamp);
            task.setTaskStatus(TaskStatus.INTERRUPT.getStatus());
            task.setProgress(0.0);

            logicService.run("taskSave", Task.convertToJson(task));

            taskIdRinBufferSequence.remove(id);
            TaskManageHandler.threadMap.remove(id);
        }
        return taskResult;
    }

}
