package com.af.v4.system.common.task.service;

import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.task.entity.Task;
import com.af.v4.system.common.task.entity.TaskHandler;
import com.af.v4.system.common.task.entity.vo.TaskResult;
import com.af.v4.system.common.task.enums.TaskStatus;
import com.af.v4.system.common.task.product.TaskProduct;
import com.af.v4.system.common.task.utils.LogicUtil;
import com.af.v4.system.common.task.utils.SerializationUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.sql.Timestamp;

@Component
public class TaskService {

    public static final String CONFIG_NAME = "taskManagement";
    public static final String SERVICE_NAME = "af-task";

    // 是否本地logic调用(af-task 项目内部调用)
    public static boolean isLocal = true;
    private final Logger LOGGER = LoggerFactory.getLogger(TaskService.class);
    private final TaskProduct taskProduct;


    @Autowired
    private RedisService redisService;
    @Autowired
    private LogicService logicService;

    public TaskService(TaskProduct taskProduct) {

        this.taskProduct = taskProduct;
    }

    /**
     * 提交任务进度
     *
     * @param id           任务id
     * @param taskProgress 任务进度 取值:(0 < taskProgress <= 1)
     */
    public void updateProgress(Integer id, String userId, double taskProgress, String taskType) {

        Boolean useWebSocket = !TaskProduct.config.has("taskType") || (!TaskProduct.config.getJSONObject("taskType").has("useWebSocket") || TaskProduct.config.getJSONObject(taskType).optBoolean("useWebSocket", false));
        if (taskProgress < 0 || taskProgress > 1)
            return;
        LogicUtil.run(logicService, "taskUpdateProgress", String.valueOf(new JSONObject().put("id", id).put("f_user_id", userId).put("f_task_progress", taskProgress).put("useWebSocket", useWebSocket)), TaskService.isLocal);
    }

    /**
     * 推送执行任务
     *
     * @param data {
     *             "f_task_name": [任务名称]
     *             "f_task_type": [任务类型]
     *             "f_task_class": [任务类class]
     *             "f_user_id": [用户id]
     *             "f_param": [param参数字段, 不做持久化存储]
     *             // 如果是从任务管理页面发起的任务, 需要传递任务执行项目
     *             "id": [id]
     *             }
     * @return 任务id
     */
    public JSONObject send(JSONObject data) throws IOException, ClassNotFoundException {

        LOGGER.info(">>>> 任务准备执行: " + data.optString("f_task_name"));
        isLocal = TaskProduct.config.getJSONObject("taskType").getJSONObject(data.getString("f_task_type")).optBoolean("isLocal", true);
        // param 存储在 redis中, 不存储在数据库
        Object f_param = data.opt("f_param");
        data.put("f_param", "");

        // 保存一条消息
        boolean sendMessage = TaskProduct.config.getJSONObject("taskType").getJSONObject(data.getString("f_task_type")).optBoolean("sendMessage");
        if (sendMessage) {
            JSONObject param = new JSONObject("""
                    {
                        "type": "task",
                        "businessType": "任务"
                    }
                    """);


            JSONObject progressJson = new JSONObject()
                    .put("taskName", data.getString("f_task_name"))
                    .put("progress", 0);
            JSONObject message = new JSONObject()
                    .put("title", data.getString("f_task_name"))
                    .put("description", "进度 0")
                    .put("messageType", "task")
                    .put("progress", progressJson);
            param.put("event", String.valueOf(message));

            param.put("userId", data.getInt("f_user_id"));

            JSONArray targetUsers = new JSONArray();
            targetUsers.put(new JSONObject().put("id", data.getInt("f_user_id")));
            param.put("targetUsers", targetUsers);

            JSONObject extraParam = new JSONObject("""
                    {
                        "path": "_/buslll任务管理",
                        "batchUpdate": true
                    }
                    """);
            param.put("extra", String.valueOf(extraParam));

            JSONObject messageRes = logicService.remoteRun(ServiceNameConstants.SYSTEM_SERVICE, "createMessage", String.valueOf(param));
            data.put("f_tag6", messageRes.getJSONArray("data").getJSONObject(0).getString("id"));
        }

        // 如果是从服务端界面发起的任务, 直接执行 [判断条件修改为, 是否传递id]
        JSONObject taskSendRes = data;
        int TTL = TaskProduct.config.getJSONObject("taskType").getJSONObject(data.getString("f_task_type")).optInt("TTL", 60 * 60 * 1000);
        // 不包含id, 首次执行任务
        if (!data.has("id")) {
            // 如果是从客户端界面发起的任务, 在进行任务保存及判断是否重复任务
            data.put("TTL", TTL);
            taskSendRes = (JSONObject) LogicUtil.run(logicService, "taskSend", data.toString(), isLocal);
            // 判断重复任务 任务状态为 执行成功, 获取的 重复任务 返回任务id
            if (taskSendRes.getBoolean("isExist")) {
                return taskSendRes;
            }
        } else {
            LogicUtil.run(logicService, "taskSave", String.valueOf(data), TaskService.isLocal);
        }
        int taskId = taskSendRes.optInt("id");
        // 携带id任务再次执行, redis中存在任务参数可以直接获取
        if (!redisService.hasKey("TASK-" + taskId + "-param")) {
            redisService.set("TASK-" + taskId + "-param", SerializationUtils.serialize(f_param), TTL);
        }
        Timestamp f_create_time = taskSendRes.has("f_create_time") && taskSendRes.get("f_create_time") != null ? Timestamp.valueOf(taskSendRes.getString("f_create_time")) : new Timestamp(System.currentTimeMillis());
        Task task = new Task(taskSendRes.getString("f_task_name"),
                f_create_time,
                TaskStatus.PENDING.getStatus(),
                taskSendRes.getString("f_task_type"),
                taskSendRes.getString("f_task_class"),
                taskSendRes.getString("f_user_id"));
        task.setId(taskSendRes.getInt("id"));
        task.setTag1(data.opt("f_tag1"));
        task.setTag2(data.opt("f_tag2"));
        task.setTag3(data.opt("f_tag3"));
        task.setTag4(data.opt("f_tag4"));
        task.setTag5(data.opt("f_tag5"));
        task.setTag6(data.opt("f_tag6"));
        // 否则 推送任务 执行
        Object bean = SpringUtils.getBean(Class.forName(task.getTaskClass()));
        TaskResult<Task> taskTaskResult = taskProduct.publishEvent(task, (TaskHandler) bean);
        return Task.convertToJson(taskTaskResult.getData());
    }

    /**
     * 中断任务
     *
     * @param data {
     *             "id": [任务id]
     *             }
     * @return 中断结果
     */
    public TaskResult<String> interrupt(JSONObject data) {

        return taskProduct.interrupt(data.getInt("id"));
    }

}
