package com.af.v4.system.common.task.consumer;

import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.task.entity.Task;
import com.af.v4.system.common.task.entity.TaskHandler;
import com.af.v4.system.common.task.entity.vo.TaskResult;
import com.af.v4.system.common.task.enums.TaskStatus;
import com.af.v4.system.common.task.event.TaskEvent;
import com.af.v4.system.common.task.product.TaskProduct;
import com.af.v4.system.common.task.service.TaskService;
import com.af.v4.system.common.task.utils.LogicUtil;
import com.lmax.disruptor.WorkHandler;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;

/**
 * 任务消息处理器
 * <p>
 * WorkHandler 处理重复消费
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class TaskManageHandler implements WorkHandler<TaskEvent> {

    private final static Logger LOGGER = LoggerFactory.getLogger(TaskManageHandler.class);

    public final static Map<Integer, Thread> threadMap = new HashMap<>();

    @Autowired
    private RedisService redisService;
    @Autowired
    private LogicService logicService;

    @Override
    public void onEvent(TaskEvent event) {

        TaskHandler taskHandler = event.getTemplateTaskHandler();
        Task task = event.getTemplateTask();
        LOGGER.info(">>>> 任务消费开始: " + task.getTaskName());

        JSONObject taskJson = ((JSONArray) LogicUtil.run(logicService, "taskGetByQuery", String.valueOf(new JSONObject()
                .put("query", "id = '" + task.getId() + "'")), TaskService.isLocal)).getJSONObject(0);
        if (TaskStatus.INTERRUPT.getStatus().equals(taskJson.getString("f_task_status"))) {
            return;
        }
        Timestamp timestamp = new Timestamp(System.currentTimeMillis());
        task.setTaskStatus(TaskStatus.RUNNING.getStatus());
        task.setStartTime(timestamp);
        task.setProgress(0.0);

        LogicUtil.run(logicService, "taskSave", String.valueOf(Task.convertToJson(task)), TaskService.isLocal);

        try {
            threadMap.put(task.getId(), Thread.currentThread());
            // 消费时, redis中不存在此任务, 返回失败
            if (!redisService.hasKey("TASK-" + task.getId() + "-param")) {
                task.setTaskStatus(TaskStatus.FAILED.getStatus());
                task.setResult("执行失败");
                throw new RuntimeException();
            } else {
                String param = redisService.get("TASK-" + task.getId() + "-param");
                task.setParam(param);
            }
            LOGGER.info(">>>> 任务开始执行 " + task.getTaskName());
            TaskResult<String> res = taskHandler.run(task, taskHandler);
            LOGGER.info(">>>> 任务执行结束, 执行结果" + res);
            if (res.getMsg().equals(TaskStatus.SUCCESS)) {
                LOGGER.info(">>>> 消费结束 删除redis数据: " + "TASK-" + task.getId() + "-param");
                redisService.delete("TASK-" + task.getId() + "-param");
            }
            task.setTaskStatus(res.getMsg().getStatus());
            task.setResult(res.getData());
        } catch (Exception e) {
            LOGGER.info(">>>> 任务消费异常" + task.getTaskName(), e);
            task.setTaskStatus(TaskStatus.FAILED.getStatus());
        } finally {
            task.setParam(null);
            TaskProduct.taskIdRinBufferSequence.remove(task.getId());
            threadMap.remove(task.getId());

            Timestamp endTime = new Timestamp(System.currentTimeMillis());
            task.setEndTime(endTime);
            task.setProgress(1.0);
            LogicUtil.run(logicService, "taskSave", String.valueOf(Task.convertToJson(task)), TaskService.isLocal);
            LOGGER.info(">>>> 任务消费结束: " + task.getTaskName());
        }

    }
}
