package com.af.v4.system.common.task.service;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.plugins.http.RestTools;
import com.af.v4.system.common.task.entity.Task;
import com.af.v4.system.common.task.entity.TaskHandler;
import com.af.v4.system.common.task.entity.vo.TaskResult;
import com.af.v4.system.common.task.enums.TaskStatus;
import com.af.v4.system.common.task.product.TaskProduct;
import com.af.v4.system.common.task.utils.InvokeUtil;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.sql.Timestamp;

@Component
public class TaskService implements InitializingBean {

    public static final String CONFIG_NAME = "taskManagement";
    public static JSONObject config;
    static Logger LOGGER = LoggerFactory.getLogger(TaskService.class);
    private final TaskProduct taskProduct;

    public TaskService(TaskProduct taskProduct, LiuLiConfigService liuLiConfigService) {
        this.taskProduct = taskProduct;
        config = liuLiConfigService.get(TaskService.CONFIG_NAME);
    }

    /**
     * 提交任务进度
     *
     * @param id           任务id
     * @param taskProgress 任务进度 取值:(0 < taskProgress <= 1)
     */
    public static void updateProgress(Integer id, String userId, double taskProgress) {
        if (taskProgress < 0 || taskProgress > 1)
            return;

        LOGGER.info(">>>> 更新进度: " + id + "-" + taskProgress);
        // 发送到服务端进行 进度变更
        RestTools.post(TaskProduct.serverAddress + "/taskUpdateProgress", new JSONObject().put("id", id).put("f_user_id", userId).put("f_task_progress", taskProgress));
    }

    /**
     * 推送执行任务
     *
     * @param data 创建新任务并执行
     *             {
     *             "f_task_name": [任务名称]
     *             "f_task_type": [任务类型]
     *             "f_task_class": [任务类class]
     *             "f_param": [任务参数 run方法 param值]
     *             "f_user_id": [用户id]
     *             // 如果是从任务管理页面发起的任务, 需要选择执行器 传递执行器 id
     *             "f_task_group_id": [执行器id]
     *             }
     * @return 任务id
     */
    public JSONObject send(JSONObject data) throws ClassNotFoundException, InvocationTargetException, NoSuchMethodException, IllegalAccessException, InstantiationException {

        LOGGER.info(">>>> 发起任务: " + data);

        // 如果是从服务端界面发起的任务, 直接执行
        JSONObject taskSendRes = data;
        if (!data.has("f_task_group_id")) {
            // 如果是从客户端界面发起的任务, 在琉璃中心获取执行器名称 address, 携带参数 发送服务端 进行任务保存及判断是否重复任务
            data.put("f_address", config.getString("address")).put("f_app_name", config.getString("appName"));
            String sendRes = RestTools.post(TaskProduct.serverAddress + "/taskSend", data);
            data = new JSONObject(sendRes).getJSONObject("data");
            taskSendRes = data.getJSONObject("data");
            // 怎么判断重复任务 任务状态为 执行成功, 获取的 重复任务 返回任务id
            if (data.getBoolean("isExist")) {
                return taskSendRes;
            }
        }

        Timestamp f_create_time = taskSendRes.has("f_create_time") ? Timestamp.valueOf(taskSendRes.getString("f_create_time")) : new Timestamp(System.currentTimeMillis());
        Task task = new Task(taskSendRes.getString("f_task_name"),
                f_create_time,
                TaskStatus.PENDING.getStatus(),
                taskSendRes.getString("f_task_type"),
                taskSendRes.getString("f_task_class"),
                taskSendRes.getString("f_param"),
                taskSendRes.getString("f_user_id"));
        task.setTaskGroupId(taskSendRes.getInt("f_task_group_id"));
        task.setId(taskSendRes.getInt("id"));
        // 否则 推送任务 执行
        TaskResult<Task> taskTaskResult = taskProduct.publishEvent(task, (TaskHandler) InvokeUtil.createInstance(task.getTaskClass()));
        return Task.convertToJson(taskTaskResult.getData());
    }

    /**
     * 中断任务
     *
     * @param data {
     *             "id": [任务id]
     *             }
     * @return 中断结果
     */
    public TaskResult<String> interrupt(JSONObject data) throws IOException, ClassNotFoundException {

        LOGGER.info(">>>> 中断任务: " + data);
        return taskProduct.interrupt(data.getInt("id"));
    }

    @Override
    public void afterPropertiesSet() {

        String register = RestTools.post(TaskProduct.serverAddress + "/taskClientRegister", new JSONObject()
                .put("address", config.getString("address"))
                .put("appName", config.getString("appName")));
        try {
            if (new JSONObject(register).getInt("code") != 200)
                throw new RuntimeException("任务服务注册失败, 未启动task服务端");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
