package com.af.v4.system.common.socket.core.server;

import com.af.v4.system.common.socket.config.SocketConfigItem;
import io.netty.bootstrap.AbstractBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * SOCKET服务端
 *
 * @param <B> {@link io.netty.bootstrap.AbstractBootstrap}的任意实现
 * @param <C> {@link io.netty.channel.Channel}的任意实现
 */
public abstract class SocketServer<B extends AbstractBootstrap<B, C>, C extends Channel> {

    protected final Logger LOGGER = LoggerFactory.getLogger(getClass());

    /**
     * 启动服务端
     *
     * @param socketConfigItem socket配置项
     */
    public void run(SocketConfigItem socketConfigItem) {
        String socketType = socketConfigItem.getType().getValue();
        LOGGER.info("Netty: 初始化{}服务端[{}],监听端口{},回调业务逻辑名[{}]", socketType, socketConfigItem.getName(), socketConfigItem.getPorts(), socketConfigItem.getLogicName());
        AbstractBootstrap<B, C> bootstrap = initBootstrap();
        // 绑定端口，开始接收进来的连接
        socketConfigItem.getPorts().forEach(port -> {
            try {
                ChannelFuture channelFuture = bootstrap.bind(Integer.parseInt(port.toString())).sync();
                channelFuture.channel().closeFuture().addListener((ChannelFutureListener) future -> {
                    //通过回调只关闭自己监听的channel
                    future.channel().close();
                });
            } catch (InterruptedException e) {
                LOGGER.error("启动{}服务端[{}]时发生异常", socketType, socketConfigItem.getName(), e);
                throw new RuntimeException(e);
            }
        });
    }

    /**
     * 初始化netty引导实例
     *
     * @return 引导实例
     */
    protected abstract AbstractBootstrap<B, C> initBootstrap();

    /**
     * 销毁实例
     */
    public abstract void destroy();
}
