package com.af.v4.system.common.socket.core.channel;

import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.proxy.logic.ILogicServiceProxy;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.socket.SocketServerManager;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.client.ClientManager;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Arrays;

public abstract class AbstractChannelHandler<T> extends SimpleChannelInboundHandler<T> {

    private final Logger LOGGER = LoggerFactory.getLogger(AbstractChannelHandler.class);

    @Override
    protected final void channelRead0(ChannelHandlerContext ctx, T msg) {
        String channelId = ctx.channel().id().asLongText();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("channelId:{}, 收到消息", channelId);
        }
        ClientManager.setChannelId(channelId);
        if (ClientManager.getChannelData() == null) {
            initChannelData(ctx);
        }
        // 执行读方法
        read(ctx, msg);
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        String channelId = ctx.channel().id().asLongText();
        SocketAddress socketAddress = ctx.channel().remoteAddress();
        String address;
        if (socketAddress == null) {
            address = "-";
        } else {
            address = socketAddress.toString();
        }
        LOGGER.info("channelId:{}, address:{}, 成功建立连接", channelId, address);
        initChannelData(ctx);
        super.channelActive(ctx);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        try {
            ctx.close();
            ClientManager.remove();
        } catch (Exception e) {
            LOGGER.error("关闭通道时发生异常", e);
        }
        super.channelInactive(ctx);
    }

    protected abstract void read(ChannelHandlerContext ctx, T msg);

    private void initChannelData(ChannelHandlerContext ctx) {
        // 存入管道信息
        Channel channel = ctx.channel();
        String port = String.valueOf(((InetSocketAddress) channel.localAddress()).getPort());
        SocketConfigItem configItem = SocketServerManager.getSocketConfigItemByPort(port);
        ClientManager.add(channel.id().asLongText(), new ChannelData(configItem, channel, new JSONObject()));
    }

    protected Object runBusiness(ByteBuf content, String remoteAddress) {
        SocketConfigItem configItem = ClientManager.getConfig();
        byte[] data = new byte[content.readableBytes()];
        content.readBytes(data);
        LOGGER.info("数据原始体：{}", Arrays.toString(data));
        String realData;
        try {
            realData = switch (configItem.getDecodeType()) {
                case HEX -> ConvertTools.byteToHexStr(data);
                default -> ConvertTools.bytesToStr(data);
            };
        } catch (Exception e) {
            LOGGER.error("数据解析失败：", e);
            return null;
        }

        JSONObject params = new JSONObject();
        params.put("value", realData);
        params.put("address", remoteAddress);

        try {
            return SpringUtils.getBean(ILogicServiceProxy.class).run(configItem.getLogicName(), params);
        } catch (Exception e) {
            if (e instanceof LogicException ex) {
                LOGGER.error("请求来源'{}',发生业务异常.\n{}", remoteAddress, ex.getStack());
            } else {
                LOGGER.error("请求来源'{}',发生服务端异常.", remoteAddress, e);
            }
            throw e;
        }
    }

}
