package com.af.v4.system.common.socket;

import com.af.v4.system.common.socket.config.SocketClientConfigItem;
import com.af.v4.system.common.socket.config.SocketConfig;
import com.af.v4.system.common.socket.core.client.modbus.ModBusClient;
import com.af.v4.system.common.socket.core.client.tcp.TcpClient;
import com.af.v4.system.common.socket.core.server.SocketClient;
import com.af.v4.system.common.socket.enums.SocketTypeEnum;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Component
public class SocketClientManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SocketClientManager.class);

    private static final Map<String, SocketClientConfigItem> socketClientConfigMap = new HashMap<>();
    private final SocketConfig socketConfig;
    private final List<SocketClient<?>> socketServerList = new ArrayList<>();

    public SocketClientManager(SocketConfig socketConfig) {
        this.socketConfig = socketConfig;
    }

    public static SocketClientConfigItem getSocketConfigItemByPort(String port) {
        return socketClientConfigMap.get(port);
    }

    /**
     * 连接socket服务端
     */
    @PostConstruct
    public void startAll() {
        List<SocketClientConfigItem> socketConfigItemList = socketConfig.getSocketServerClientList();
        if (socketConfigItemList != null) {
            socketConfigItemList.forEach(item -> {
                SocketClient<?> socketServer = createSocketClient(item);
                socketServerList.add(socketServer);
                socketClientConfigMap.put(item.getAddress(), item);
                if (socketServer != null) {
                    socketServer.connect(item);
                }
            });
        }
    }

    /**
     * 创建SocketClient实例
     */
    private SocketClient<?> createSocketClient(SocketClientConfigItem item) {
        SocketClient<?> socketClient = null;
        if (item.getType() == SocketTypeEnum.TCP) {
            socketClient = new TcpClient();
        } else if (item.getType() == SocketTypeEnum.MODBUS_TCP) {
            socketClient = new ModBusClient();
        }
        return socketClient;
    }

    @PreDestroy
    public void destroyAll() {
        for (SocketClient<?> socketServer : socketServerList) {
            socketServer.destroy();
        }
    }
}
