package com.af.v4.system.common.socket.core.channel.impl;

import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.client.ChannelManager;
import io.netty.channel.Channel;
import io.netty.channel.ChannelId;
import io.netty.handler.codec.mqtt.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.af.v4.system.common.socket.core.client.ChannelManager.*;
import static io.netty.handler.codec.mqtt.MqttMessageType.*;
import static io.netty.handler.codec.mqtt.MqttQoS.AT_LEAST_ONCE;

/**
 * @Description
 * @Author Eraser
 */

public class MqttMsgBack {

    private static final Logger log = LoggerFactory.getLogger(MqttMsgBack.class);


    /**
     * 确认连接请求
     *
     * @param channel
     * @param mqttMessage
     * @param configItem
     * @param exist        是否已存在连接
     */
    public void connack(Channel channel, MqttMessage mqttMessage, SocketConfigItem configItem, boolean exist) {
        MqttConnectMessage mqttConnectMessage = (MqttConnectMessage) mqttMessage;
        MqttFixedHeader mqttFixedHeaderInfo = mqttConnectMessage.fixedHeader();
        MqttConnectVariableHeader mqttConnectVariableHeaderInfo = mqttConnectMessage.variableHeader();
        // 获取连接参数
        String connectUsername = mqttConnectMessage.payload().userName();
        String connectPassword = new String(mqttConnectMessage.payload().passwordInBytes(), StandardCharsets.UTF_8);
        // 验证连接参数
        MqttConnectReturnCode connectionAccepted;
        if (exist) {
            //	在一个网络连接上，客户端只能发送一次CONNECT报文。服务端必须将客户端发送的第二个CONNECT报文当作协议违规处理并断开客户端的连接
            connectionAccepted = MqttConnectReturnCode.CONNECTION_REFUSED_BANNED;
            disconnack(channel);
        } else if (configItem.getUsername().equals(connectUsername) && configItem.getPassword().equals(connectPassword)){
            // 成功
            connectionAccepted = MqttConnectReturnCode.CONNECTION_ACCEPTED;
            log.info("设备上线,[ channelId:{},clientId:{}", channel.id(), mqttConnectMessage.payload().clientIdentifier() + "]");
            ChannelManager.add(channel.id(), channel);
        } else {
            // 失败
            connectionAccepted = MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USERNAME_OR_PASSWORD;
            log.info("设备连接失败,请检查用户名和密码!");
        }
        //	构建返回报文， 可变报头
        MqttConnAckVariableHeader mqttConnAckVariableHeaderBack =
                new MqttConnAckVariableHeader(connectionAccepted,
                        mqttConnectVariableHeaderInfo.isCleanSession());
        //	构建返回报文， 固定报头
        MqttFixedHeader mqttFixedHeaderBack = new MqttFixedHeader(MqttMessageType.CONNACK,
                mqttFixedHeaderInfo.isDup(), MqttQoS.AT_MOST_ONCE, mqttFixedHeaderInfo.isRetain(), 0x02);
        //	构建CONNACK消息体
        MqttConnAckMessage connAck = new MqttConnAckMessage(mqttFixedHeaderBack, mqttConnAckVariableHeaderBack);
        channel.writeAndFlush(connAck);
    }

    public void disconnack(Channel channel) {
        log.info("设备下线,channelId：{}", channel.id());
        MQTTremoveChannel(channel.id());
    }

    /**
     * 根据qos发布确认
     *
     * @param channel
     * @param mqttMessage
     */
    public void puback(Channel channel, MqttMessage mqttMessage) {
        MqttPublishMessage mqttPublishMessage = (MqttPublishMessage) mqttMessage;
        MqttFixedHeader mqttFixedHeaderInfo = mqttPublishMessage.fixedHeader();
        MqttQoS qos = mqttFixedHeaderInfo.qosLevel();
        //注意：	readableBytes会改变写指针位置，使后续推送数据时，读取数据为空，需要重置	读指针
        byte[] headBytes = new byte[mqttPublishMessage.payload().readableBytes()];
        mqttPublishMessage.payload().readBytes(headBytes);
        String data = new String(headBytes);
        //重置读取的指针
        mqttPublishMessage.payload().resetReaderIndex();
        switch (qos) {
            case AT_MOST_ONCE:        //	至多一次
                //推送到订阅的客户端
                subscribSend(mqttMessage);
                break;
            case AT_LEAST_ONCE:        //	至少一次
                //	构建返回报文， 可变报头
                MqttMessageIdVariableHeader mqttMessageIdVariableHeaderBack =
                        MqttMessageIdVariableHeader.from(mqttPublishMessage.variableHeader().packetId());
                //	构建返回报文， 固定报头
                MqttFixedHeader mqttFixedHeaderBack = new MqttFixedHeader(PUBACK,
                        mqttFixedHeaderInfo.isDup(), MqttQoS.AT_MOST_ONCE, mqttFixedHeaderInfo.isRetain(), 0x02);
                //	构建PUBACK消息体
                MqttPubAckMessage pubAck = new MqttPubAckMessage(mqttFixedHeaderBack, mqttMessageIdVariableHeaderBack);
                log.info("back--" + pubAck.toString());
                channel.writeAndFlush(pubAck);
                //推送到订阅的客户端
                subscribSend(mqttMessage);
                break;
            case EXACTLY_ONCE:        //	刚好一次
                //	构建返回报文， 固定报头
                MqttFixedHeader mqttFixedHeaderBack2 = new MqttFixedHeader(PUBREC, false,
                        AT_LEAST_ONCE, false, 0x02);
                //	构建返回报文， 可变报头
                MqttMessageIdVariableHeader mqttMessageIdVariableHeaderBack2 =
                        MqttMessageIdVariableHeader.from(mqttPublishMessage.variableHeader().packetId());
                MqttMessage mqttMessageBack = new MqttMessage(mqttFixedHeaderBack2, mqttMessageIdVariableHeaderBack2);
                //服务端收到publis的QoS2的消息之后，服务端需要保存一个msgid的记录，并且进入一个状态，
                // 即之后不管来了几个这个msgid的消息，都不管他，认为是重复的，丢弃。
                //接收到publish的QoS2消息之后，不能马上投递给上层，而是在本地做持久化，将消息保存起来。
                int mqttMessageId = mqttPublishMessage.variableHeader().packetId();
                if (!mqttMessageIdMap.containsKey(mqttMessageId)) {
                    //不存在此消息，将此消息暂存 //todo 这里可以换成redis做缓存
                    mqttMessageIdMap.put(mqttMessageId, mqttMessage);
                    log.info("消息ID" + mqttMessageId + "-->Qos2级别消息，消息缓存");
                } else {
                    //重复发送消息，直接返回
                    log.info(mqttPublishMessage.variableHeader().packetId() + "消息重复：" + mqttPublishMessage.fixedHeader().isDup());
                    return;
                }
                mqttPublishMessage.retain();
                channel.writeAndFlush(mqttMessageBack);
                break;
            default:
                break;
        }
    }

    /**
     * 发布完成 qos2
     *
     * @param channel
     * @param mqttMessage
     */
    public void pubcomp(Channel channel, MqttMessage mqttMessage) {
        MqttMessageIdVariableHeader messageIdVariableHeader =
                (MqttMessageIdVariableHeader) mqttMessage.variableHeader();
        //	构建返回报文， 固定报头
        MqttFixedHeader mqttFixedHeaderBack = new MqttFixedHeader(MqttMessageType.PUBCOMP, false,
                MqttQoS.AT_MOST_ONCE, false, 0x02);
        //	构建返回报文， 可变报头
        MqttMessageIdVariableHeader mqttMessageIdVariableHeaderBack =
                MqttMessageIdVariableHeader.from(messageIdVariableHeader.messageId());
        MqttMessage mqttMessageBack = new MqttMessage(mqttFixedHeaderBack, mqttMessageIdVariableHeaderBack);
        channel.writeAndFlush(mqttMessageBack);
    }

    /**
     * 订阅确认
     *
     * @param channel
     * @param mqttMessage
     */
    public void suback(Channel channel, MqttMessage mqttMessage) {
        MqttSubscribeMessage mqttSubscribeMessage = (MqttSubscribeMessage) mqttMessage;
        MqttMessageIdVariableHeader messageIdVariableHeader = mqttSubscribeMessage.variableHeader();
        //	构建返回报文， 可变报头
        MqttMessageIdVariableHeader variableHeaderBack =
                MqttMessageIdVariableHeader.from(messageIdVariableHeader.messageId());
        Set<String> topics =
                mqttSubscribeMessage.payload().topicSubscriptions().stream().map(MqttTopicSubscription::topicName).collect(Collectors.toSet());
        //log.info(topics.toString());
        List<Integer> grantedQoSLevels = new ArrayList<>(topics.size());
        for (int i = 0; i < topics.size(); i++) {
            grantedQoSLevels.add(mqttSubscribeMessage.payload().topicSubscriptions().get(i).qualityOfService().value());
        }
        //	构建返回报文	有效负载
        MqttSubAckPayload payloadBack = new MqttSubAckPayload(grantedQoSLevels);
        //	构建返回报文	固定报头
        MqttFixedHeader mqttFixedHeaderBack = new MqttFixedHeader(MqttMessageType.SUBACK, false, MqttQoS.AT_MOST_ONCE
                , false, 2 + topics.size());
        //	构建返回报文	订阅确认
        MqttSubAckMessage subAck = new MqttSubAckMessage(mqttFixedHeaderBack, variableHeaderBack, payloadBack);
        channel.writeAndFlush(subAck);
    }

    /**
     * 取消订阅确认
     *
     * @param channel
     * @param mqttMessage
     */
    public void unsuback(Channel channel, MqttMessage mqttMessage) {
        MqttMessageIdVariableHeader messageIdVariableHeader =
                (MqttMessageIdVariableHeader) mqttMessage.variableHeader();
        //	构建返回报文	可变报头
        MqttMessageIdVariableHeader variableHeaderBack =
                MqttMessageIdVariableHeader.from(messageIdVariableHeader.messageId());
        //	构建返回报文	固定报头
        MqttFixedHeader mqttFixedHeaderBack = new MqttFixedHeader(MqttMessageType.UNSUBACK, false,
                MqttQoS.AT_MOST_ONCE, false, 2);
        //	构建返回报文	取消订阅确认
        MqttUnsubAckMessage unSubAck = new MqttUnsubAckMessage(mqttFixedHeaderBack, variableHeaderBack);
        channel.writeAndFlush(unSubAck);
    }

    /**
     * 心跳响应
     *
     * @param channel
     * @param mqttMessage
     */
    public void pingresp(Channel channel, MqttMessage mqttMessage) {
        //	心跳响应报文	11010000 00000000  固定报文
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PINGRESP, false, MqttQoS.AT_MOST_ONCE,
                false, 0);
        MqttMessage mqttMessageBack = new MqttMessage(fixedHeader);
        channel.writeAndFlush(mqttMessageBack);
    }

    /**
     * 添加订阅
     *
     * @param channel
     * @param mqttMessage
     */
    public void sub(Channel channel, MqttMessage mqttMessage) {
        MqttSubscribePayload SubscribePayload = (MqttSubscribePayload) mqttMessage.payload();
        for (int i = 0; i < SubscribePayload.topicSubscriptions().size(); i++) {
            String topicname = SubscribePayload.topicSubscriptions().get(i).topicName();
            boolean tag = subscribeMap.containsKey(topicname);
            if (tag) {
                List<ChannelId> channelIds = subscribeMap.get(topicname);
                if (!channelIds.contains(channel.id())) {
                    channelIds.add(channel.id());
                } else {
                    log.warn(channel.id() + "重复订阅");
                }
                subscribeMap.put(topicname, channelIds);
            } else {
                List<ChannelId> channelIds = new ArrayList<>();
                channelIds.add(channel.id());
                subscribeMap.put(topicname, channelIds);
            }
            log.info(channel.id() + "订阅地址————》" + topicname);
        }
    }

    /**
     * 取消订阅
     *
     * @param channel
     * @param mqttMessage
     */
    public void unSub(Channel channel, MqttMessage mqttMessage) {
        Object Unsubscribe = mqttMessage.payload();
        MqttUnsubscribePayload unsubscribePayload = (MqttUnsubscribePayload) Unsubscribe;
        int len = unsubscribePayload.topics().size();
        for (int i = 0; i < len; i++) {
            String topicname = unsubscribePayload.topics().get(i);
            boolean tag = subscribeMap.containsKey(topicname);
            if (tag) {
                List<ChannelId> channelIds = subscribeMap.get(topicname);
                channelIds.remove(channel.id());
                subscribeMap.put(topicname, channelIds);
            } else {
                log.error("不存在订阅地址——>" + topicname);
            }
            log.info(channel.id() + "取消订阅地址————》" + topicname);
        }
    }

    /**
     * 订阅推送
     */
    public void subscribSend(MqttMessage mqttMessage) {
        MqttPublishMessage mqttPublishMessage = (MqttPublishMessage) mqttMessage;
        Object obj = mqttMessage.variableHeader();
        MqttPublishVariableHeader variableHeader = (MqttPublishVariableHeader) obj;
        String topicName = variableHeader.topicName();
        int packetId = variableHeader.packetId();
        //固定消息头 注意此处的消息类型PUBLISH mqtt协议
        MqttFixedHeader FixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, false, AT_LEAST_ONCE,
                false, 0);
        //可变消息头
        MqttPublishVariableHeader mqttPublishVariableHeader = new MqttPublishVariableHeader(topicName, packetId);
        //推送消息体
        MqttPublishMessage mqttPublishMessageResult = new MqttPublishMessage(FixedHeader, mqttPublishVariableHeader,
                mqttPublishMessage.content());
        log.info("推送地址————》" + topicName);
        Set<ChannelId> set = new HashSet();
        // 防止重复发送
        subscribeMap.entrySet().forEach(item -> {
            if (matches(item.getKey(), topicName)) {
                set.addAll(item.getValue());
            }
        });
        set.forEach(item -> {
            //订阅次此topic的Mqtt客户端搜到此消息，
            Channel channelSub = ChannelManager.getChannelMap().get(item.toString());
            //writeAndFlush会将ByteBuf的引用释放，refCnt会减去1,使用retain加1
            if (channelSub != null) {
                mqttPublishMessageResult.retain();
                channelSub.writeAndFlush(mqttPublishMessageResult);
            }
        });
    }

    /**
     * 判断一个主题是否与带有通配符的主题模式匹配。
     *
     * @param wildcardTopic 带有通配符的主题模式
     * @param topic         需要匹配的主题
     * @return 如果匹配则返回 true，否则返回 false。
     */
    public boolean matches(String wildcardTopic, String topic) {
        if (wildcardTopic.equals("#")) {
            return true; // # 通配符匹配任意主题
        }

        String[] wildcardParts = wildcardTopic.split("/");
        String[] topicParts = topic.split("/");

        if (wildcardParts.length != topicParts.length && !wildcardTopic.endsWith("#")) {
            return false; // 如果不是 # 结尾并且长度不同，则不匹配
        }

        for (int i = 0; i < wildcardParts.length; i++) {
            if ("+".equals(wildcardParts[i])) {
                continue; // + 通配符匹配单个单词
            } else if ("#".equals(wildcardParts[i])) {
                return true; // # 通配符匹配剩余的所有单词
            } else if (!wildcardParts[i].equals(topicParts[i])) {
                return false; // 不匹配
            }
        }

        return true;
    }
}

