package com.af.v4.system.common.socket.core.channel.impl;

import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.channel.AbstractChannelHandler;
import com.af.v4.system.common.socket.core.client.ChannelManager;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.*;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.af.v4.system.common.socket.core.client.ChannelManager.*;

/**
 * @Description
 * @Author Eraser
 */

@ChannelHandler.Sharable
public class MqttServerChannelHandler extends AbstractChannelHandler {

    private final Logger log = LoggerFactory.getLogger(MqttServerChannelHandler.class);

    private final SocketConfigItem configItem;

    public MqttServerChannelHandler(SocketConfigItem configItem) {
        this.configItem = configItem;
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        if (cause instanceof ClassCastException) {
            log.error("消息处理过程中发生类型转换异常: {}", cause.getMessage());
        } else if (cause instanceof IllegalArgumentException) {
            log.error("发生非法参数异常: {}", cause.getMessage());
        } else {
            log.error("下游出现意外异常。", cause);
        }
        ctx.close();
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, Object msg) {
        if (null != msg) {
            MqttMsgBack mqttMsgBack = new MqttMsgBack();
            MqttMessage mqttMessage = (MqttMessage) msg;
            log.info("info--" + mqttMessage);
            MqttFixedHeader mqttFixedHeader = mqttMessage.fixedHeader();
            Channel channel = ctx.channel();
            switch (mqttFixedHeader.messageType()) {
                case CONNECT:        // 客户端连接
                    // 对客户端进行认证管理等
                    mqttMsgBack.connack(channel, mqttMessage,configItem,ChannelManager.getChannelMap().contains(channel));
                    break;
                case PUBLISH:        //	客户端发布消息
                    //	PUBACK报文是对QoS 1等级的PUBLISH报文的响应
                    mqttMsgBack.puback(channel, mqttMessage);
                    String topic = ((MqttPublishVariableHeader)mqttMessage.variableHeader()).topicName();
                    JSONObject address = new JSONObject()
                            .put("topic", topic)
                            .put("remoteAddress", channel.remoteAddress().toString());
                    runBusiness((ByteBuf) mqttMessage.payload(), address.toString());
                    break;
                // PUBREL	Qos2级别消息，客户端返回
                case PUBREL:
                    //	PUBREL（客户端发给服务端）报文是对PUBREC(服务端发给客户端)报文的响应
                    //服务端收到pubrel之后，正式将消息投递给上层应用层。
                    MqttMessageIdVariableHeader VariableHeader =
                            (MqttMessageIdVariableHeader) mqttMessage.variableHeader();
                    if (mqttMessageIdMap.containsKey(VariableHeader.messageId())) {
                        log.warn("移除消息缓存-->消息id" + VariableHeader.messageId());
                        mqttMsgBack.subscribSend(mqttMessageIdMap.get(VariableHeader.messageId()));
                        mqttMsgBack.pubcomp(channel, mqttMessage);
                        mqttMessageIdMap.remove(VariableHeader.messageId());
                    } else {
                        //后续多次收到REL消息，制作comp响应
                        mqttMsgBack.pubcomp(channel, mqttMessage);
                    }
                    break;
                case SUBSCRIBE:        //	客户端订阅主题
                    //	客户端向服务端发送SUBSCRIBE报文用于创建一个或多个订阅，每个订阅注册客户端关心的一个或多个主题。
                    //	为了将应用消息转发给与那些订阅匹配的主题，服务端发送PUBLISH报文给客户端。
                    //	SUBSCRIBE报文也（为每个订阅）指定了最大的QoS等级，服务端根据这个发送应用消息给客户端
                    mqttMsgBack.suback(channel, mqttMessage);
                    mqttMsgBack.sub(channel, mqttMessage);
                    break;
                case UNSUBSCRIBE:    //	客户端取消订阅
                    //	客户端发送UNSUBSCRIBE报文给服务端，用于取消订阅主题
                    mqttMsgBack.unsuback(channel, mqttMessage);
                    mqttMsgBack.unSub(channel, mqttMessage);
                    break;
                case PINGREQ:        //	客户端发起心跳
                    //	客户端发送PINGREQ报文给服务端的
                    //	在没有任何其它控制报文从客户端发给服务的时，告知服务端客户端还活着
                    //	请求服务端发送 响应确认它还活着，使用网络以确认网络连接没有断开
                    mqttMsgBack.pingresp(channel, mqttMessage);
                    break;
                case DISCONNECT:    //	客户端主动断开连接
                    mqttMsgBack.disconnack(channel);
                    break;
                default:
                    break;
            }
        }
    }

}
