package com.af.v4.system.common.socket.config;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.socket.enums.MsgTypeEnum;
import com.af.v4.system.common.socket.enums.SocketTypeEnum;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * socket配置
 *
 * @author Mr.river
 */
@Component
public class SocketConfig {
    private static final String CONFIG_NAME = "SocketConfig";
    private static final String CLIENT_CONFIG_NAME = "SocketClientConfig";
    private final Logger LOGGER = LoggerFactory.getLogger(SocketConfig.class);
    /**
     * Socket服务端列表
     */
    private List<SocketConfigItem> socketConfigItemList;
    /**
     * Socket客户端列表
     */
    private List<SocketClientConfigItem> socketClientConfigItemList;

    public SocketConfig(LiuLiConfigService liuLiConfigService) {
        JSONObject config = liuLiConfigService.get(CONFIG_NAME);
        if (config != null) {
            JSONArray socketServerList = config.optJSONArray("server");
            if (socketServerList != null) {
                socketConfigItemList = parseSocketConfigItems(socketServerList);
            }
        }
        JSONObject client_config = liuLiConfigService.get(CLIENT_CONFIG_NAME);
        if (client_config != null) {
            JSONArray socketClientList = client_config.optJSONArray("server");
            if (socketClientList != null) {
                socketClientConfigItemList = parseSocketClientConfigItems(socketClientList);
            }
        }
    }

    private List<SocketConfigItem> parseSocketConfigItems(JSONArray socketServerList) {
        List<SocketConfigItem> list = new ArrayList<>(socketServerList.length());
        socketServerList.forEach(item -> {
            JSONObject itemObj = (JSONObject) item;
            list.add(new SocketConfigItem()
                    .setName(itemObj.getString("name"))
                    .setType(SocketTypeEnum.toType(itemObj.getString("type")))
                    .setEncodeType(MsgTypeEnum.toType(itemObj.getString("encodeType")))
                    .setDecodeType(MsgTypeEnum.toType(itemObj.getString("decodeType")))
                    .setUsername(itemObj.optString("username"))
                    .setPassword(itemObj.optString("password"))
                    .setPorts(itemObj.getJSONArray("ports"))
                    .setLogicName(itemObj.getString("logicName"))
            );
        });
        return list;
    }

    private List<SocketClientConfigItem> parseSocketClientConfigItems(JSONArray socketClientList) {
        List<SocketClientConfigItem> list = new ArrayList<>(socketClientList.length());
        socketClientList.forEach(item -> {
            try {
                JSONObject itemObj = (JSONObject) item;
                if (itemObj.optBoolean("enable", false)) {
                    list.add(new SocketClientConfigItem()
                            .setName(itemObj.getString("name"))
                            .setType(SocketTypeEnum.toType(itemObj.getString("type")))
                            .setEncodeType(MsgTypeEnum.toType(itemObj.optString("encodeType", "NORMAL")))
                            .setDecodeType(MsgTypeEnum.toType(itemObj.optString("decodeType", "NORMAL")))
                            .setPorts(itemObj.getInt("port"))
                            .setHost(itemObj.getString("host"))
                            .setConnectionTimeout(itemObj.optInt("connectionTimeout", 10))
                            .setReadTimeout(itemObj.optInt("readTimeout", 30))
                            .setEvents(itemObj.optJSONArray("events") == null ? new ArrayList<>() :
                                    parseModbusEvents(itemObj.optJSONArray("events")))
                    );
                }
            } catch (JSONException e) {
                LOGGER.error("客户端信息添加失败，错误信息：{}", e.toString());
            }
        });
        return list;
    }

    private List<ClientEvent> parseModbusEvents(JSONArray events) {
        List<ClientEvent> list = new ArrayList<>(events.length());
        for (Object item : events) {
            try {
                JSONObject itemObj = (JSONObject) item;
                int functionCode = itemObj.getInt("functionCode");
                if (functionCode != 3 && functionCode != 1) {
                    LOGGER.error("modBus事件初始化失败，错误信息：暂只支持功能吗 1(读线圈)，3（读保持寄存器）");
                    continue;
                }
                list.add(new ClientEvent()
                        .setRemark(itemObj.getString("remark"))
                        .setAddress(itemObj.getString("address"))
                        .setFunctionCode(functionCode)
                        .setLength(itemObj.getInt("length"))
                        .setAlias(itemObj.getString("alias"))
                        .setTime(itemObj.optInt("time", 3))
                        .setDecodeType(MsgTypeEnum.toType(itemObj.optString("decodeType", "NORMAL")))
                        .setLogicName(itemObj.getString("logicName"))
                );
            } catch (JSONException e) {
                LOGGER.error("modBus事件初始化失败，错误信息：{}", e.toString());
            }
        }
        return list;
    }

    public List<SocketConfigItem> getSocketServerList() {
        return socketConfigItemList;
    }

    public List<SocketClientConfigItem> getSocketServerClientList() {
        return socketClientConfigItemList;
    }
}
