package com.af.v4.system.common.socket.core.server.mqtt.message;

import com.af.v4.system.common.socket.core.channel.mqtt.MqttMsgBack;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.handler.codec.mqtt.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;

public class MqttUtils {

    private static final Logger log = LoggerFactory.getLogger(MqttUtils.class);

    // 记录消息id的变量，id值范围1~65535
    private final AtomicInteger nextMessageId = new AtomicInteger(1);
    private static final ReentrantLock LOCK = new ReentrantLock();

    /**
     * 主动发送消息
     *
     * @param topic   :主题名称
     * @param message ：消息体
     * @param qos     ： 服务质量等级
     * @param retain  ：
     *                true:表示发送的消息需要一直持久保存（不受服务器重启影响），不但要发送给当前的订阅者，并且以后新来的订阅了此Topic name的订阅者会马上得到推送。
     *                false:仅仅为当前订阅者推送此消息。
     */
    public void publish(String topic, String message, int qos, Boolean retain) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, false, MqttQoS.valueOf(qos), retain, 0);
        MqttPublishVariableHeader variableHeader = new MqttPublishVariableHeader(topic, getNewMessageId().messageId());
        // 创建 ByteBuf 分配器
        ByteBufAllocator allocator = ByteBufAllocator.DEFAULT;
        // 将字符串转换为 ByteBuf
        ByteBuf byteBuf = allocator.buffer();
        byteBuf.writeCharSequence(message, StandardCharsets.UTF_8);
        MqttPublishMessage mqttPublishMessage = new MqttPublishMessage(fixedHeader, variableHeader, byteBuf);
        //将消息发送给订阅的客户端
        new MqttMsgBack().subscribSend(mqttPublishMessage);
    }

    /**
     * 功能描述: 获取消息id，int数，从1开始不能大于65535
     *
     * @return io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader
     */
    private MqttMessageIdVariableHeader getNewMessageId() {
        int messageId;
        LOCK.lock();
        try {
            nextMessageId.compareAndSet(0xffff, 1);
            messageId = nextMessageId.getAndIncrement();
            return MqttMessageIdVariableHeader.from(messageId);
        } finally {
            LOCK.unlock();
        }
    }
}
