package com.af.v4.system.common.socket.core.client;

import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.socket.config.SocketConfigItem;
import com.af.v4.system.common.socket.core.channel.ChannelData;
import io.netty.channel.Channel;
import io.netty.util.HashedWheelTimer;
import org.json.JSONObject;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 客户端管理器
 */
public class ClientManager {
    private static final Map<String, ChannelData> clientChannels = new ConcurrentHashMap<>();
    private static final ThreadLocal<String> channelIdLocal = new ThreadLocal<>();
    private static volatile HashedWheelTimer TIMER;

    private static HashedWheelTimer getTimer() {
        if (TIMER == null) {
            synchronized (ClientManager.class) {
                if (TIMER == null) {
                    TIMER = new HashedWheelTimer();
                }
            }
        }

        return TIMER;
    }

    public static void setChannelId(String channelId) {
        channelIdLocal.set(channelId);
    }

    public static void add(String channelId, ChannelData channelData) {
        clientChannels.put(channelId, channelData);
    }

    public static void remove() {
        String channelId = channelIdLocal.get();
        if (channelId != null) {
            clientChannels.remove(channelId);
            channelIdLocal.remove();
        }
    }

    public static void delayedSend(String message, Integer sleepTime) {
        getTimer().newTimeout((timeout) -> send(message), Long.parseLong(sleepTime.toString()), TimeUnit.MILLISECONDS);
    }

    public static void send(String message) {
        String channelId = channelIdLocal.get();
        ChannelData channelData = clientChannels.get(channelId);
        Channel channel = channelData.channel();
        byte[] sendMsgBytes;
        if (channel != null) {
            sendMsgBytes = switch (channelData.configItem().getEncodeType()) {
                case BASE64 -> ConvertTools.base64Decode(message.getBytes());
                case HEX -> ConvertTools.hexStrToByte(message);
                case HEX_BASE64 -> ConvertTools.base64Decode(ConvertTools.hexStrToByte(message));
                default -> message.getBytes();
            };
            channel.writeAndFlush(channel.alloc().buffer().writeBytes(sendMsgBytes));
        }
    }

    public static SocketConfigItem getConfig() {
        return getChannelData().configItem();
    }

    public static ChannelData getChannelData() {
        String channelId = channelIdLocal.get();
        return clientChannels.get(channelId);
    }

    /**
     * 设置全局参数
     *
     * @param key   参数key
     * @param value 值
     */
    public static void setGlobalParams(String key, Object value) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.globalParams();
        globalParamsItem.put(key, value);
    }

    /**
     * 获取全局参数
     *
     * @param key 参数key
     */
    public static Object getGlobalParams(String key) {
        ChannelData channelData = ClientManager.getChannelData();
        JSONObject globalParamsItem = channelData.globalParams();
        return globalParamsItem.opt(key);
    }
}
