package com.af.v4.system.common.socket.core.server;

import com.af.v4.system.common.socket.config.SocketClientConfigItem;
import com.af.v4.system.common.socket.core.client.ChannelManager;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetSocketAddress;

/**
 * 连接SOCKET服务端
 *
 * @param <C> {@link Channel}的任意实现
 */
public abstract class SocketClient<C extends Channel> extends Bootstrap {

    private final Logger LOGGER = LoggerFactory.getLogger(SocketClient.class);

    /**
     * 启动服务端
     *
     * @param socketConfigItem socket配置项
     */
    public void connect(SocketClientConfigItem socketConfigItem) {
        String socketType = socketConfigItem.getType().getValue();
        LOGGER.info("开始连接{}服务端[{}],监听端口{}", socketType, socketConfigItem.getName(), socketConfigItem.getAddress());
        Bootstrap bootstrap = initBootstrap(socketConfigItem);
        // 绑定端口，开始接收进来的连接
        try {
            bootstrap.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, socketConfigItem.getConnectionTimeout() * 1000); // 设置连接超时时间
            ChannelFuture channelFuture = bootstrap.connect(new InetSocketAddress(socketConfigItem.getHost(), socketConfigItem.getPort())).sync();
            if (channelFuture.isSuccess()) {
                ChannelManager.add(socketConfigItem, channelFuture.channel());
                senMes(socketConfigItem);
                channelFuture.channel().closeFuture().addListener((ChannelFutureListener) future -> {
                    //通过回调只关闭自己监听的channel
                    future.channel().close();
                });
            }
        } catch (InterruptedException e) {
            LOGGER.error("启动{}服务端[{}]时发生异常", socketType, socketConfigItem.getName(), e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 初始化netty引导实例
     *
     * @return 引导实例
     */
    protected abstract Bootstrap initBootstrap(SocketClientConfigItem socketClientConfigItem);

    /**
     * 销毁实例
     */
    public abstract void destroy();

    /**
     * 连接成功之后发送消息
     */
    public abstract void senMes(SocketClientConfigItem socketClientConfigItem);

    /**
     * 是否为linux环境
     */
    protected boolean isLinux() {
        String os = System.getProperty("os.name").toLowerCase();
        return os.contains("linux");
    }
}
