package com.af.v4.system.common.socket.core.client.modbus;

import com.af.v4.system.common.socket.config.ClientEvent;
import com.af.v4.system.common.socket.config.SocketClientConfigItem;
import com.af.v4.system.common.socket.core.channel.ChannelData;
import com.af.v4.system.common.socket.core.channel.impl.ModbusRequestDecoder;
import com.af.v4.system.common.socket.core.channel.impl.ModbusResponseEncoder;
import com.af.v4.system.common.socket.core.channel.impl.ModbusTCPChannelHandler;
import com.af.v4.system.common.socket.core.client.ClientManager;
import com.af.v4.system.common.socket.core.server.SocketClient;
import com.af.v4.system.common.socket.core.server.modbus.message.ModbusSendUtil;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PreDestroy;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;


/**
 * TCP客户端
 */
public class ModBusClient extends SocketClient<ServerChannel> {
    private final Logger LOGGER = LoggerFactory.getLogger(ModBusClient.class);

    // 当同一个 chanel 读取多个 modbus 寄存器时 只返回数据不返回地址，因此用事务标识符区分
    public static final ConcurrentHashMap<Short, ClientEvent> modBusEventMap = new ConcurrentHashMap<>(128);

    private EventLoopGroup group;

    @Override
    protected Bootstrap initBootstrap(SocketClientConfigItem socketConfigItem) {
        Class<? extends Channel> channelClass;
        if (isLinux()) {
            group = new EpollEventLoopGroup();
            channelClass = EpollSocketChannel.class;
        } else {
            group = new NioEventLoopGroup();
            channelClass = NioSocketChannel.class;
        }

        return new Bootstrap()
                .group(group)
                .channel(channelClass)
                .handler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) {
                        ch.pipeline().addLast(new ModbusRequestDecoder());
                        ch.pipeline().addLast(new ModbusResponseEncoder());
                        ch.pipeline().addLast(new ModbusTCPChannelHandler());
                    }
                });
    }

    @PreDestroy
    public void destroy() {
        if (group != null) {
            group.shutdownGracefully();
        }
    }

    @Override
    public void senMes(SocketClientConfigItem socketClientConfigItem) {
        if (!socketClientConfigItem.getEvents().isEmpty()) {
            // 创建定时器池
            int taskCount = (int) Math.ceil(socketClientConfigItem.getEvents().size() / 2.0) + 1;
            ScheduledExecutorService executor = Executors.newScheduledThreadPool(taskCount);
            // 组织任务列表
            List<Runnable> tasks = socketClientConfigItem.getEvents().stream()
                    .map(event -> (Runnable) () -> {
                        // 通过 地址和长度 算出 2 位 数字 十六位标识符
                        short transactionIdentifier = ModbusSendUtil.getTransactionIdentifier(event.getAddress(),  event.getLength());
                        modBusEventMap.put(transactionIdentifier,event);
                        event.setTransactionId(transactionIdentifier);
                        // 开始组织读取数据
                        ModbusSendUtil.readData(socketClientConfigItem.getAddress(), event);
                    }).toList();
            for (int i = 0; i < tasks.size(); i++) {
                Runnable task = tasks.get(i);
                // 获取定时执行时间开始执行任务
                long interval = socketClientConfigItem.getEvents().get(i).getTime();
                executor.scheduleAtFixedRate(task, 0, interval, TimeUnit.SECONDS);
            }
        }
    }
}