package com.af.v4.system.common.security.service;

import cn.hutool.crypto.SecureUtil;
import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.utils.JwtUtils;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.core.utils.ip.IpUtils;
import com.af.v4.system.common.core.utils.uuid.IdUtils;
import com.af.v4.system.common.plugins.core.SecureTools;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.security.utils.SecurityUtils;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * token验证处理
 */
@Component
public class TokenService {
    protected static final long MILLIS_SECOND = 1000;
    protected static final long MILLIS_MINUTE = 60 * MILLIS_SECOND;
    private static final Long MILLIS_MINUTE_TEN = CacheConstants.REFRESH_TIME * MILLIS_MINUTE;
    private static final long EXPIRE_TIME = CacheConstants.EXPIRATION;

    private static final String ACCESS_TOKEN = CacheConstants.LOGIN_TOKEN_KEY;
    private static final String SESSION_SECURE_KEY = CacheConstants.SESSION_SECURE_KEY;

    private static final String RsaPublicKey = CacheConstants.RSA_PUBLIC_KEY;


    private final RedisService redisService;

    public TokenService(RedisService redisService) {
        this.redisService = redisService;
    }

    /**
     * 创建令牌
     */
    public Map<String, Object> createToken(LoginUser loginUser) {
        String token = IdUtils.fastUUID();
        Long userId = loginUser.getSysUser().getUserId();
        String userName = loginUser.getSysUser().getUserName();
        loginUser.setToken(token);
        loginUser.setUserid(userId);
        loginUser.setUsername(userName);
        loginUser.setIpaddr(IpUtils.getIpAddr());
        loginUser.setSessionEncryptKey(SecureUtil.sha256(token).substring(0, 32));
        refreshToken(loginUser);

        // Jwt存储信息
        Map<String, Object> claimsMap = new HashMap<>(3);
        claimsMap.put(SecurityConstants.USER_KEY, token);
        claimsMap.put(SecurityConstants.DETAILS_USER_ID, userId);
        claimsMap.put(SecurityConstants.DETAILS_USERNAME, userName);

        // 接口返回信息
        Map<String, Object> rspMap = new HashMap<>(3);
        rspMap.put("access_token", JwtUtils.createToken(claimsMap));
        rspMap.put("expires_in", EXPIRE_TIME);
        rspMap.put("session", SecureTools.RSAEncrypt(loginUser.getSessionEncryptKey(), RsaPublicKey));
        rspMap.put("resources", loginUser.getResources());
        return rspMap;
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser() {
        return getLoginUser(ServletUtils.getRequest());
    }

    /**
     * 设置用户身份信息
     */
    public void setLoginUser(LoginUser loginUser) {
        if (StringUtils.isNotNull(loginUser) && StringUtils.isNotEmpty(loginUser.getToken())) {
            refreshToken(loginUser);
        }
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser(HttpServletRequest request) {
        // 获取请求携带的令牌
        String token = SecurityUtils.getToken(request);
        return getLoginUser(token);
    }

    /**
     * 获取用户身份信息
     *
     * @return 用户信息
     */
    public LoginUser getLoginUser(String token) {
        if (StringUtils.isNotEmpty(token)) {
            String userKey = JwtUtils.getUserKey(token);
            return redisService.get(getTokenKey(userKey));
        }
        return null;
    }

    /**
     * 删除用户缓存信息
     */
    public void delLoginUser(String token) {
        if (StringUtils.isNotEmpty(token)) {
            String userKey = JwtUtils.getUserKey(token);
            redisService.delete(getTokenKey(userKey));
            redisService.delete(getSecureKey(userKey));
        }
    }

    /**
     * 验证令牌有效期，相差不足120分钟，自动刷新缓存
     *
     * @param loginUser 登录信息
     */
    public void verifyToken(LoginUser loginUser) {
        long expireTime = loginUser.getExpireTime();
        long currentTime = System.currentTimeMillis();
        if (expireTime - currentTime <= MILLIS_MINUTE_TEN) {
            refreshToken(loginUser);
        }
    }

    /**
     * 刷新令牌有效期
     *
     * @param loginUser 登录信息
     */
    public void refreshToken(LoginUser loginUser) {
        loginUser.setLoginTime(System.currentTimeMillis());
        loginUser.setExpireTime(loginUser.getLoginTime() + EXPIRE_TIME * MILLIS_MINUTE);
        // 根据uuid将loginUser缓存
        String userKey = getTokenKey(loginUser.getToken());
        redisService.set(userKey, loginUser, EXPIRE_TIME * 60);
        // 根据uuid缓存登录安全信息
        String secureKey = getSecureKey(loginUser.getToken());
        redisService.set(secureKey, loginUser.getSessionEncryptKey(), EXPIRE_TIME * 60);
    }

    private String getTokenKey(String token) {
        return ACCESS_TOKEN + token;
    }

    private String getSecureKey(String token) {
        return SESSION_SECURE_KEY + token;
    }
}
