package com.af.v4.system.common.security.filter;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.context.SecurityContextHolder;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.security.auth.AuthUtil;
import com.af.v4.system.common.security.utils.SecurityUtils;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.security.SignatureException;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 自定义请求头过滤器，将Header数据封装到线程变量中方便获取
 * 注意：此过滤器会同时验证当前用户有效期自动刷新有效期
 */
public class ScopedUserContextFilter extends OncePerRequestFilter {

    private static final Logger LOGGER = LoggerFactory.getLogger(ScopedUserContextFilter.class);

    @Override
    protected void doFilterInternal(@NonNull HttpServletRequest request,
                                    @NonNull HttpServletResponse response,
                                    @NonNull FilterChain filterChain) {

        String userIdValue = ServletUtils.getHeader(request, SecurityConstants.DETAILS_USER_ID);
        String userNameValue = ServletUtils.getHeader(request, SecurityConstants.DETAILS_USERNAME);
        String userKeyValue = ServletUtils.getHeader(request, SecurityConstants.USER_KEY);

        Map<String, Object> userContextMap = new ConcurrentHashMap<>(6);
        userContextMap.put(SecurityConstants.DETAILS_USER_ID, userIdValue == null ? StringUtils.EMPTY : userIdValue);
        userContextMap.put(SecurityConstants.DETAILS_USERNAME, userNameValue == null ? StringUtils.EMPTY : userNameValue);
        userContextMap.put(SecurityConstants.USER_KEY, userKeyValue == null ? StringUtils.EMPTY : userKeyValue);

        String token = SecurityUtils.getToken();
        if (StringUtils.isNotEmpty(token)) {
            try {
                LoginUser loginUser = AuthUtil.getLoginUser(token);
                if (StringUtils.isNotNull(loginUser)) {
                    AuthUtil.verifyLoginUserExpire(loginUser);
                    userContextMap.put(SecurityConstants.LOGIN_USER, loginUser);
                }
            } catch (SignatureException | MalformedJwtException e) {
                // 该错误常见于JWT签名与本地计算的签名不匹配，为了保证V3兼容性，此处需要把异常过滤
            } catch (JwtException e) {
                LOGGER.error("JWT 解析失败", e);
            }
        }

        // 用 ScopedValue 包裹整个请求
        ScopedValue.where(SecurityContextHolder.getScopedUserInfo(), userContextMap).run(() -> {
            try {
                filterChain.doFilter(request, response);
            } catch (IOException | ServletException e) {
                throw new RuntimeException(e);
            }
        });
    }
}
