package com.af.v4.system.common.security.interceptor;

import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.context.SecurityContextHolder;
import com.af.v4.system.common.core.utils.ServletUtils;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.security.auth.AuthUtil;
import com.af.v4.system.common.security.utils.SecurityUtils;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.security.SignatureException;
import jakarta.annotation.Nonnull;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.lang.NonNullApi;
import org.springframework.lang.Nullable;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.AsyncHandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 自定义请求头拦截器，将Header数据封装到线程变量中方便获取
 * 注意：此拦截器会同时验证当前用户有效期自动刷新有效期
 */
public class HeaderInterceptor implements AsyncHandlerInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(HeaderInterceptor.class);

    @Override
    public void postHandle(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler, @Nullable ModelAndView modelAndView) throws Exception {
        String userIdValue = ServletUtils.getHeader(request, SecurityConstants.DETAILS_USER_ID);
        String userNameValue = ServletUtils.getHeader(request, SecurityConstants.DETAILS_USERNAME);
        String userKeyValue = ServletUtils.getHeader(request, SecurityConstants.USER_KEY);

        Map<String, Object> map = new ConcurrentHashMap<>(6);
        map.put(SecurityConstants.DETAILS_USER_ID, userIdValue == null ? StringUtils.EMPTY : userIdValue);
        map.put(SecurityConstants.DETAILS_USERNAME, userNameValue == null ? StringUtils.EMPTY : userNameValue);
        map.put(SecurityConstants.USER_KEY, userKeyValue == null ? StringUtils.EMPTY : userKeyValue);

        String token = SecurityUtils.getToken();
        if (StringUtils.isNotEmpty(token)) {
            LoginUser loginUser;
            try {
                loginUser = AuthUtil.getLoginUser(token);
            } catch (SignatureException | MalformedJwtException e) {
                // 该错误常见于JWT签名与本地计算的签名不匹配，为了保证V3兼容性，此处需要把异常过滤
                loginUser = null;
            } catch (JwtException e) {
                LOGGER.error("校验出错", e);
                loginUser = null;
            }
            if (StringUtils.isNotNull(loginUser)) {
                AuthUtil.verifyLoginUserExpire(loginUser);
                map.put(SecurityConstants.LOGIN_USER, loginUser);
            }
        }
        ScopedValue.where(SecurityContextHolder.getScopedUserInfo(), map).run(
                () -> {
                    try {
                        AsyncHandlerInterceptor.super.postHandle(request, response, handler, modelAndView);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }
        );
    }
}
