package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.resource.enums.ResourceType;
import org.dom4j.Element;
import org.json.JSONObject;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.stereotype.Service;

import java.lang.annotation.Annotation;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 业务插件映射器
 *
 * @author Mr.river
 */
@Component
public class PluginMapper extends AbstractResourceMapper<PluginMapper.PluginResource> {

    private static Map<String, Object> pluginMap;

    protected PluginMapper(ModuleMapper moduleMapper, ApplicationUtils applicationUtils, LiuLiConfigService liuLiConfigService) {
        super(moduleMapper, applicationUtils, liuLiConfigService);
    }

    public static Map<String, Object> getPluginMap() {
        return pluginMap;
    }

    @Override
    protected void loadMap() {
        super.loadMap();
        pluginMap = getAllMap().entrySet().stream()
                .filter(object -> object.getValue().getObj() != null) // 跳过null值
                .collect(Collectors.toMap(Map.Entry::getKey, object -> object.getValue().getObj()));
    }

    @Override
    public ResourceType getResType() {
        return ResourceType.PLUGIN;
    }

    @Override
    public String getFileName() {
        return "plugins.xml";
    }

    @Override
    public String getFolderName() {
        return null;
    }

    @Override
    protected boolean isBinaryResource() {
        return true;
    }

    @Override
    public boolean isSupportCompile() {
        return false;
    }

    @Override
    protected String getResPath(Element element, String moduleName, String parentModuleName) {
        return element.attribute("class").getValue();
    }

    @Override
    protected PluginMapper.PluginResource buildResourceImpl(String moduleName, String key, String path, JSONObject liuliValue, Map<String, Object> attrMap) {
        // 插件类路径
        Object className = attrMap.get("class");

        return new PluginMapper.PluginResource(
                this,
                moduleName,
                key,
                path,
                getObjByClassName(className.toString(), moduleName)
        );
    }

    private Object getObjByClassName(String className, String moduleName) {
        try {
            Class<?> c = Class.forName(className);
            Object obj;
            // 如果是spring的bean，让spring取
            if (isSpringBean(c)) {
                obj = SpringUtils.getBean(c);
            } else {
                obj = c.getDeclaredConstructor().newInstance();
            }
            return obj;
        } catch (ClassNotFoundException | NoClassDefFoundError ex) {
            if (moduleName == null) {
                moduleName = "/";
            }
            LOGGER.warn("未找到插件, 所属模块: {}, 路径: {}", moduleName, ex.getMessage());
            return null;
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    private boolean isSpringBean(Class<?> c) {
        for (Annotation annotation : c.getAnnotations()) {
            if (annotation.annotationType() == Component.class ||
                    annotation.annotationType() == Service.class ||
                    annotation.annotationType() == Controller.class
            ) {
                return true;
            }
        }
        return false;
    }

    /**
     * 插件资源
     */
    public static class PluginResource extends CommonResource {
        /**
         * 插件对象
         */
        private final Object obj;

        public PluginResource(AbstractResourceMapper<?> mapper, String moduleName, String key, String path, Object obj) {
            super(mapper, moduleName, key, path, null);
            this.obj = obj;
        }

        public Object getObj() {
            return obj;
        }
    }
}
