package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.plugins.io.IOTools;
import com.af.v4.system.common.resource.config.ResourceConfig;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.lang.annotation.Annotation;
import java.util.*;

/**
 * 抽象资源映射器
 */
@Component
public abstract class AbstractResourceMapper<T> {
    /**
     * 组织目录名称
     */
    public static final String ORGANIZATIONAL_DIRECTORY_NAME = "organizations";
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractResourceMapper.class);
    /**
     * 系统定制版本值
     */
    private static String VERSION_ALIAS;
    private final ModuleMapper moduleMapper;
    /**
     * 资源列表
     */
    protected Map<String, Map<String, T>> map;

    protected AbstractResourceMapper(ModuleMapper moduleMapper, ResourceConfig resourceConfig) {
        this.moduleMapper = moduleMapper;
        VERSION_ALIAS = Optional.ofNullable(resourceConfig.getVersionAlias()).orElse(CacheConstants.SYSTEM_DEFAULT_VALUE);
    }

    /**
     * 获取系统定制版本别名
     *
     * @return 系统定制版本别名
     */
    public static String getVersionAliasValue() {
        return VERSION_ALIAS;
    }

    /**
     * 获取资源类型
     *
     * @return 资源类型
     */
    protected abstract String getResType();

    /**
     * 获取资源文件名
     *
     * @return 资源文件名
     */
    protected abstract String getFileName();

    /**
     * 获取资源文件夹名
     *
     * @return 资源文件夹名
     */
    protected abstract String getFolderName();

    /**
     * 加载资源列表
     */
    @PostConstruct
    protected void loadMap() {
        if (map == null) {
            map = new HashMap<>(36);
            putMapByRoot();
            moduleMapper.getMap().forEach((key, value) -> {
                String name = value.get("name");
                String path = value.get("path");
                try {
                    putMapByModule(name, path);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
    }

    /**
     * 加载根目录资源到列表
     */
    protected void putMapByRoot() {
        putMapByModule(null, null);
    }

    /**
     * 加载模块资源到列表
     *
     * @param name   模块名
     * @param parent 父模块名
     */
    protected void putMapByModule(String name, String parent) {
        String str;
        String fileName = getFileName();
        String resType = getResType();
        if (parent == null) {
            if (name == null) {
                str = fileName;
            } else {
                str = name + "/" + fileName;
            }
        } else {
            str = parent + "/" + name + "/" + fileName;
        }
        IOTools.getStream(str, stream -> {
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(stream);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
            Element root = document.getRootElement();
            for (Iterator<Element> item = root.elementIterator(resType); item.hasNext(); ) {
                Element element = item.next();
                appendRes(element, name, parent);
            }
        });
    }

    /**
     * 获取资源Map
     *
     * @return 全部资源
     */
    public Map<String, Map<String, T>> getAllMap() {
        return map;
    }

    /**
     * 根据Key获取值
     *
     * @param key KEY
     * @return value
     */
    public CommonResource<T> getValue(String key) {
        String alias = getAlias(key);
        if (map.containsKey(alias)) {
            return new CommonResource<>(alias, map.get(alias).get("value"));
        }
        return new CommonResource<>(alias, null);
    }

    /**
     * 根据key获取资源别名
     *
     * @param key key
     * @return 资源别名
     */
    protected String getAlias(String key) {
        String versionAliasValue = getVersionAliasValue();
        // 如果是标准产品，直接返回
        if (versionAliasValue.equals(CacheConstants.SYSTEM_DEFAULT_VALUE)) {
            return key;
        }
        String alias = key + "@" + versionAliasValue;
        return map.containsKey(alias) ? alias : key;
    }

    /**
     * 根据Key获取对应资源Map
     *
     * @param key KEY
     * @return 资源Map
     */
    public Map<String, T> getRes(String key) {
        String alias = getAlias(key);
        return map.getOrDefault(alias, null);
    }

    /**
     * 追加资源
     *
     * @param element          XML元素
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    protected Map<String, T> appendRes(Element element, String moduleName, String parentModuleName) {
        String alias = element.attribute("alias").getValue();
        // 如果是分公司资源，修改注册的别名
        if (Objects.equals(parentModuleName, ORGANIZATIONAL_DIRECTORY_NAME)) {
            alias = alias + "@" + moduleName;
        }
        // 验证别名是否重复
        if (map.containsKey(alias)) {
            String resType = getResType();
            throw new ServiceException(resType + "别名[" + alias + "]已存在", HttpStatus.CONFIG_ERROR);
        }
        // 设置属性
        Map<String, T> res = new HashMap<>(4);
        T value;
        Attribute pathAttribute = element.attribute("path");
        if (pathAttribute != null) {
            String path = pathAttribute.getValue();
            String folderName = getFolderName();
            path = (parentModuleName == null ?
                    moduleName + "/" + folderName + "/" :
                    parentModuleName + "/" + moduleName + "/" + folderName + "/") + path;
            value = (T) path;
        } else {
            String className = element.attribute("class").getValue();
            Object obj = getClassByName(className, moduleName);
            value = (T) obj;
        }
        if (value != null) {
            res.put("alias", (T) alias);
            res.put("value", value);
            map.put(alias, res);
        }
        return res;
    }

    private Object getClassByName(String className, String moduleName) {
        try {
            Class<?> c = Class.forName(className);
            Object obj;
            // 如果是spring的bean，让spring取
            if (isSpringBean(c)) {
                obj = SpringUtils.getBean(c);
            } else {
                obj = c.getDeclaredConstructor().newInstance();
            }
            return obj;
        } catch (ClassNotFoundException | NoClassDefFoundError ex) {
            LOGGER.warn("未找到类, 模块=" + moduleName + ", 类=" + ex.getMessage());
            return null;
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    private boolean isSpringBean(Class<?> c) {
        for (Annotation annotation : c.getAnnotations()) {
            if (annotation.annotationType() == Component.class ||
                    annotation.annotationType() == Service.class ||
                    annotation.annotationType() == RestController.class
            ) {
                return true;
            }
        }
        return false;
    }

    /**
     * 资源结果集
     *
     * @param <T> 资源数据类型
     */
    public static class CommonResource<T> {
        private final String alias;
        private final T path;


        CommonResource(String alias, T path) {
            this.alias = alias;
            this.path = path;
        }


        public String getAlias() {
            return alias;
        }

        public T getPath() {
            return path;
        }
    }
}
