package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.plugins.io.IOTools;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.File;
import java.net.URL;
import java.util.*;

/**
 * 模块映射器
 *
 * @author Mr.river
 */
@Component
public class ModuleMapper {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModuleMapper.class);

    /**
     * 模块映射文件名称
     */
    private static final String MODULE_FILE_NAME = "module.xml";

    /**
     * 模块属性：名称
     */
    private static final String MODE_NAME = "module";

    /**
     * 模块属性：导入
     */
    private static final String IMPORT_NAME = "import";

    /**
     * 基础租户目录路径
     */
    private static final String BASE_TENANT_PATH = "/" + AbstractResourceMapper.TENANT_MODULE_NAME + "/";

    private static Map<String, Map<String, String>> map;

    public ModuleMapper() {
        map = new LinkedHashMap<>(8);
        loadModuleBootstrap(MODULE_FILE_NAME, true);
    }

    /**
     * 获取所有模块映射文件
     */
    public Map<String, Map<String, String>> getMap() {
        return map;
    }

    /**
     * 遍历租户模块
     */
    private void parseTenantModule() {
        URL url = ModuleMapper.class.getResource(BASE_TENANT_PATH);
        if (url != null) {
            File directory = new File(url.getPath());
            File[] directoryList = directory.listFiles();
            if (directoryList != null) {
                for (File item : directoryList) {
                    if (item.isDirectory()) {
                        String moduleName = item.getName();
                        // 设置module属性
                        Map<String, String> module = new HashMap<>(3);
                        module.put("path", AbstractResourceMapper.TENANT_MODULE_NAME);
                        map.put(moduleName, module);
                        LOGGER.info(">>> 模块：[" + moduleName + "]加载完成");
                    }
                }
            }
        }
    }

    /**
     * 遍历依赖模块
     *
     * @param root 根节点
     */
    private void parseImportModule(Element root) {
        List<Element> ite = root.elements(IMPORT_NAME);
        if (!ite.isEmpty()) {
            for (Element elm : ite) {
                String name = elm.attribute("name").getValue();
                LOGGER.info("====从模块集[" + name + "]导入模块====");
                loadModuleBootstrap(name + "/" + MODULE_FILE_NAME, false);
            }
        }
    }

    /**
     * 遍历项目模块
     */
    private void parseProjectModule(Element root) {
        for (Iterator<Element> it = root.elementIterator(MODE_NAME); it.hasNext(); ) {
            Element elm = it.next();
            String name = elm.attribute("name").getValue();
            String upload = elm.attributeValue("upload");
            String path = elm.attributeValue("path");
            // 设置module属性
            Map<String, String> module = new HashMap<>(3);
            module.put("upload", upload);
            module.put("path", path);
            map.put(name, module);
            LOGGER.info(">>> 模块：[" + name + "]加载完成");
        }
    }

    private void loadModuleBootstrap(String path, boolean isRoot) {
        IOTools.getStream(path, stream -> {
            try {
                SAXReader reader = new SAXReader();
                Document document = reader.read(stream);
                Element root = document.getRootElement();
                parseImportModule(root);
                if (isRoot) {
                    LOGGER.info("====从租户目录导入模块====");
                    parseTenantModule();
                    LOGGER.info("====从本地目录导入模块====");
                }
                parseProjectModule(root);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
        }, errorPath -> {
            throw new ServiceException("模块定义文件[module.xml]未找到，相关路径：" + errorPath, HttpStatus.CONFIG_ERROR);
        });
    }
}
