package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.resource.core.ResourceIndexer;
import com.af.v4.system.common.resource.enums.ResourceType;
import org.dom4j.Element;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.util.Optional;

/**
 * logic映射器
 *
 * @author Mr.river
 */
@Component
public class LogicMapper extends AbstractResourceMapper<LogicMapper.LogicResource> {

    private static final Boolean DEFAULT_CACHE_STATE = false;
    private static final Long DEFAULT_CACHE_TIME = 60L;
    private static final Boolean DEFAULT_AUTH_STATE = true;
    private static final Boolean DEFAULT_MOBILE_STATE = false;
    private static final Boolean DEFAULT_CALLBACK_API_STATE = false;
    private final LiuLiConfigService liuLiConfigService;

    protected LogicMapper(ModuleMapper moduleMapper, ApplicationUtils applicationUtils, LiuLiConfigService liuLiConfigService) {
        super(moduleMapper, applicationUtils);
        this.liuLiConfigService = liuLiConfigService;
    }

    @Override
    public ResourceType getResType() {
        return ResourceType.LOGIC;
    }

    @Override
    public String getFileName() {
        return "logic.xml";
    }

    @Override
    public String getFolderName() {
        return "logics";
    }

    public LogicMapper.LogicResource getValue(String alias) {
        LogicResource resource = super.getValue(alias);
        if (Optional.ofNullable(resource).isEmpty()) {
            JSONObject liuliContent = liuLiConfigService.get(alias);
            if (liuliContent != null) {
                JSONObject globalProp = liuliContent.getJSONObject("globalProp");
                resource = new LogicResource(
                        globalProp.optString("module", null),
                        alias,
                        ResourceIndexer.DEFAULT_CLOUD_DATA_PATH_VALUE,
                        liuliContent.optString("dataSource", null),
                        liuliContent.optString("cache", null),
                        liuliContent.optString("time", null),
                        liuliContent.optString("cacheKey", null),
                        liuliContent.optString("auth", null),
                        liuliContent.optString("mobile", null),
                        liuliContent.optString("callbackApi", null)
                );
            }
        }
        return resource;
    }

    @Override
    protected LogicResource buildResource(String moduleName, String key, Object value, Element element) {
        // 所属数据源
        String dataSource = element.attributeValue("dataSource");
        // 是否缓存
        String cache = element.attributeValue("cache");
        // 自定义缓存Key
        String cacheKey = element.attributeValue("cacheKey");
        // 缓存时间
        String time = element.attributeValue("time");
        // 是否鉴权
        String auth = element.attributeValue("auth");
        // 是否支持手机端
        String mobile = element.attributeValue("mobile");
        // 是否为第三方回调接口
        String callbackApi = element.attributeValue("callbackApi");

        return new LogicResource(
                moduleName,
                key,
                value.toString(),
                dataSource,
                cache,
                time,
                cacheKey,
                auth,
                mobile,
                callbackApi
        );
    }

    /**
     * Logic资源
     */
    public static class LogicResource extends CommonResource implements MobileResourceSupport {
        /**
         * 所属数据源
         */
        private final String dataSource;
        /**
         * 是否缓存
         */
        private final boolean cache;
        /**
         * 缓存时间
         */
        private final long time;
        /**
         * 自定义缓存Key
         */
        private final String cacheKey;
        /**
         * 是否鉴权
         */
        private final boolean auth;
        /**
         * 是否支持手机端
         */
        private final boolean mobile;
        /**
         * 是否为第三方回调接口，该值为是时，响应体不进行统一封装处理
         */
        private final boolean callbackApi;

        public LogicResource(String moduleName, String key, Object path, String dataSource, String cache, String time, String cacheKey, String auth, String mobile, String callbackApi) {
            super(moduleName, key, path);
            this.dataSource = dataSource;
            this.cache = cache == null ? DEFAULT_CACHE_STATE : Boolean.parseBoolean(cache);
            this.time = time == null ? DEFAULT_CACHE_TIME : Long.parseLong(time);
            this.cacheKey = cacheKey;
            this.auth = auth == null ? DEFAULT_AUTH_STATE : Boolean.parseBoolean(auth);
            this.mobile = mobile == null ? DEFAULT_MOBILE_STATE : Boolean.parseBoolean(mobile);
            this.callbackApi = callbackApi == null ? DEFAULT_CALLBACK_API_STATE : Boolean.parseBoolean(callbackApi);
        }

        public String getDataSource() {
            return dataSource;
        }

        public boolean isCache() {
            return cache;
        }

        public long getTime() {
            return time;
        }

        public String getCacheKey() {
            return cacheKey;
        }

        public boolean getAuth() {
            return auth;
        }

        public boolean isMobile() {
            return mobile;
        }

        public boolean isCallbackApi() {
            return callbackApi;
        }
    }
}
