package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.constant.CacheConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.plugins.io.IOTools;
import com.af.v4.system.common.resource.enums.ResourceType;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.lang.annotation.Annotation;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 抽象资源映射器
 */
@Component
public abstract class AbstractResourceMapper<T extends AbstractResourceMapper.CommonResource> {
    /**
     * 租户模块名称
     */
    public static final String TENANT_MODULE_NAME = "tenants";
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractResourceMapper.class);
    private static final boolean IS_JUNIT_TEST;

    static {
        IS_JUNIT_TEST = isJUnitTest();
    }

    private final ApplicationUtils applicationUtils;
    private final ModuleMapper moduleMapper;
    /**
     * 资源列表
     */
    protected Map<String, T> map;

    protected AbstractResourceMapper(ModuleMapper moduleMapper, ApplicationUtils applicationUtils) {
        this.moduleMapper = moduleMapper;
        this.applicationUtils = applicationUtils;
    }

    /**
     * 是否为junit环境
     *
     * @return 判断结果
     */
    private static boolean isJUnitTest() {
        for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
            if (element.getClassName().startsWith("org.junit.")) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取资源类型
     *
     * @return 资源类型
     */
    public abstract ResourceType getResType();

    /**
     * 获取资源文件名
     *
     * @return 资源文件名
     */
    public abstract String getFileName();

    /**
     * 获取资源文件夹名
     *
     * @return 资源文件夹名
     */
    public abstract String getFolderName();

    /**
     * 加载资源列表
     */
    @PostConstruct
    protected void loadMap() {
        if (map == null) {
            map = new LinkedHashMap<>(36);
            putMapByRoot();
            moduleMapper.getMap().forEach((key, value) -> {
                String path = value.get("path");
                try {
                    putMapByModule(key, path);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
        }
    }

    /**
     * 加载根目录资源到列表
     */
    protected void putMapByRoot() {
        putMapByModule(null, null);
    }

    /**
     * 加载模块资源到列表
     *
     * @param name   模块名
     * @param parent 父模块名
     */
    protected void putMapByModule(String name, String parent) {
        String str;
        String fileName = getFileName();
        ResourceType resType = getResType();
        if (parent == null) {
            if (name == null) {
                str = fileName;
            } else {
                str = name + "/" + fileName;
            }
        } else {
            str = parent + "/" + name + "/" + fileName;
        }
        IOTools.getStream(str, stream -> {
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(stream);
            } catch (DocumentException e) {
                throw new RuntimeException(e);
            }
            Element root = document.getRootElement();
            for (Iterator<Element> item = root.elementIterator(resType.getValue()); item.hasNext(); ) {
                Element element = item.next();
                appendRes(element, name, parent);
            }
        });
    }

    /**
     * 获取资源Map
     *
     * @return 全部资源
     */
    public Map<String, T> getAllMap() {
        return map;
    }

    /**
     * 根据Key获取值
     *
     * @param key KEY
     * @return value
     */
    public T getValue(String key) {
        String alias = getAlias(key);
        if (map.containsKey(alias)) {
            return map.get(key);
        }
        return null;
    }

    /**
     * 根据key获取资源别名
     *
     * @param key key
     * @return 资源别名
     */
    protected String getAlias(String key) {
        String tenantName = applicationUtils.getTenantName();
        // 如果是标准产品，直接返回
        if (tenantName.equals(CacheConstants.TENANT_DEFAULT_VALUE) ||
                tenantName.equals(CacheConstants.TENANT_TEST_VALUE)) {
            return key;
        }
        String alias = key + "@" + tenantName;
        return map.containsKey(alias) ? alias : key;
    }

    /**
     * 追加资源
     *
     * @param element          XML元素
     * @param moduleName       模块名
     * @param parentModuleName 父级模块名
     */
    protected void appendRes(Element element, String moduleName, String parentModuleName) {
        if (moduleName == null) {
            moduleName = "";
        }
        String key = element.attribute("alias").getValue();
        ResourceType resType = getResType();
        // 如果是单元测试环境，验证模块是否为单元测试模块，如果是，资源别名增加test_前缀
        if (IS_JUNIT_TEST) {
            if (moduleName.startsWith("test_")) {
                key = "test_" + key;
            }
        }
        // 如果是租户资源，修改注册的别名
        if (Objects.equals(parentModuleName, TENANT_MODULE_NAME)) {
            key = key + "@" + moduleName;
        }
        // 验证别名是否重复
        if (map.containsKey(key)) {
            throw new ServiceException(resType + "别名[" + key + "]已存在", HttpStatus.CONFIG_ERROR);
        }
        // 设置属性
        Object value;
        Attribute pathAttribute = element.attribute("path");
        if (pathAttribute != null) {
            String path = pathAttribute.getValue();
            String folderName = getFolderName();
            path = (parentModuleName == null ?
                    moduleName + "/" + folderName + "/" :
                    parentModuleName + "/" + moduleName + "/" + folderName + "/") + path;
            value = path;
        } else {
            String className = element.attribute("class").getValue();
            value = getClassByName(className, moduleName);
        }
        if (value != null) {
            map.put(key, buildResource(moduleName, key, value, element));
        }
    }

    protected T buildResource(String moduleName, String key, Object value, Element element) {
        return (T) new CommonResource(moduleName, key, value);
    }

    private Object getClassByName(String className, String moduleName) {
        try {
            Class<?> c = Class.forName(className);
            Object obj;
            // 如果是spring的bean，让spring取
            if (isSpringBean(c)) {
                obj = SpringUtils.getBean(c);
            } else {
                obj = c.getDeclaredConstructor().newInstance();
            }
            return obj;
        } catch (ClassNotFoundException | NoClassDefFoundError ex) {
            if (moduleName == null) {
                moduleName = "/";
            }
            LOGGER.warn("未找到类, 模块=" + moduleName + ", 类=" + ex.getMessage());
            return null;
        } catch (RuntimeException ex) {
            throw ex;
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }
    }

    private boolean isSpringBean(Class<?> c) {
        for (Annotation annotation : c.getAnnotations()) {
            if (annotation.annotationType() == Component.class ||
                    annotation.annotationType() == Service.class ||
                    annotation.annotationType() == RestController.class
            ) {
                return true;
            }
        }
        return false;
    }

    /**
     * 移动端资源支持
     */
    public interface MobileResourceSupport {

        boolean isMobile();
    }

    /**
     * 资源结果集
     */
    public static class CommonResource {
        /**
         * 所属模块名
         */
        private final String moduleName;
        private final String alias;
        private final Object path;


        CommonResource(String moduleName, String alias, Object path) {
            this.moduleName = moduleName;
            this.alias = alias;
            this.path = path;
        }


        public String getAlias() {
            return alias;
        }

        public Object getPath() {
            return path;
        }

        public String getModuleName() {
            return moduleName;
        }
    }
}
