package com.af.v4.system.common.resource.core;

import cn.hutool.cache.Cache;
import cn.hutool.cache.CacheUtil;
import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.ApplicationUtils;
import com.af.v4.system.common.plugins.io.IOTools;
import com.af.v4.system.common.resource.enums.ResourceType;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;

/**
 * 资源索引器
 *
 * @author Mr.river
 */
@Component
public final class ResourceIndexer {

    /**
     * Logic源缓存
     */
    private static Cache<String, String> LOGIC_CACHE;
    /**
     * SQL源缓存
     */
    private static Cache<String, String> SQL_CACHE;

    private static EnvType envType = null;

    public ResourceIndexer(ApplicationUtils applicationUtils) {
        envType = applicationUtils.getEnvType();
    }

    /**
     * Logic缓存清理
     */
    public static void clearLogicCache() {
        LOGIC_CACHE.clear();
    }

    /**
     * SQL缓存清理
     */
    public static void clearSqlCache() {
        SQL_CACHE.clear();
    }

    /**
     * 获取资源文件里，指定全路径文件的文本内容，并缓存
     *
     * @param filePath : 以'/'开始的资源文件的全路径
     * @return 文件的文本内容
     */
    public static String getString(ResourceType resourceType, String alias, String filePath) {
        String source = null;
        switch (resourceType) {
            case LOGIC -> {
                if (envType == EnvType.PROD || envType == EnvType.PREVIEW) {
                    source = LOGIC_CACHE.get(alias);
                }
                if (source == null) {
                    source = IOTools.readText(filePath);
                    // 处理回车换行
                    source = source.replace("\r\n", "\n");
                    if (envType == EnvType.PROD || envType == EnvType.PREVIEW) {
                        LOGIC_CACHE.put(alias, source);
                    }
                }
                return source;
            }
            case SQL -> {
                if (envType == EnvType.PROD || envType == EnvType.PREVIEW) {
                    source = SQL_CACHE.get(alias);
                }
                if (source == null) {
                    source = IOTools.readText(filePath);
                    // 获取编译后的sql语句
                    source = source.replace("\r\n", "\n");
                    source = "$" + source;
                    //如果使用了新格式化参数写法，需要替换为解析语法树支持的写法
                    if (source.contains("${")) {
                        source = source.replace("${", "{");
                    }
                    if (envType == EnvType.PROD || envType == EnvType.PREVIEW) {
                        SQL_CACHE.put(alias, source);
                    }
                }
                return source;
            }
            default -> throw new ServiceException("错误的缓存数据类型");
        }
    }

    @PostConstruct
    public void init() {
        if (envType == EnvType.PROD || envType == EnvType.PREVIEW) {
            LOGIC_CACHE = CacheUtil.newLRUCache(100);
            SQL_CACHE = CacheUtil.newLRUCache(100);
        }
    }
}
