package com.af.v4.system.common.resource.mapper;

import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.enums.ResourceType;
import com.af.v4.system.plugins.io.IOTools;
import org.springframework.stereotype.Component;

import java.util.concurrent.ConcurrentHashMap;

/**
 * 资源映射器
 *
 * @author Mr.river
 */
@Component
public record ResourceMapper(RedisService redisService) {

    /**
     * Logic源缓存
     */
    private static final ConcurrentHashMap<String, String> CACHE_LOGIC_MAP;
    /**
     * SQL源缓存
     */
    private static final ConcurrentHashMap<String, String> CACHE_SQL_MAP;
    /**
     * Prompt源缓存
     */
    private static final ConcurrentHashMap<String, String> CACHE_PROMPT_MAP;
    public static CACHE_MODE mode = CACHE_MODE.NORMAL;

    static {
        CACHE_LOGIC_MAP = new ConcurrentHashMap<>(10);
        CACHE_SQL_MAP = new ConcurrentHashMap<>(10);
        CACHE_PROMPT_MAP = new ConcurrentHashMap<>(10);
    }

    /**
     * Logic缓存清理
     */
    public static void clearLogicCache() {
        CACHE_LOGIC_MAP.clear();
    }

    /**
     * SQL缓存清理
     */
    public static void clearSqlCache() {
        CACHE_SQL_MAP.clear();
    }

    /**
     * Prompt缓存清理
     */
    public static void clearPromptCache() {
        CACHE_PROMPT_MAP.clear();
    }

    /**
     * 获取资源文件里，指定全路径文件的文本内容，并缓存
     *
     * @param filePath : 以'/'开始的资源文件的全路径
     * @return 文件的文本内容
     */
    public String getString(ResourceType resourceType, String alias, String filePath) {
        String source;
        switch (resourceType) {
            case LOGIC -> {
                if (mode == CACHE_MODE.NORMAL && CACHE_LOGIC_MAP.containsKey(alias)) {
                    return CACHE_LOGIC_MAP.get(alias);
                } else {
                    source = IOTools.readText(filePath);
                    // 处理回车换行
                    source = source.replace("\r\n", "\n");
                    if (mode == CACHE_MODE.NORMAL) {
                        CACHE_LOGIC_MAP.put(alias, source);
                    }
                    return source;
                }
            }
            case SQL -> {
                if (mode == CACHE_MODE.NORMAL && CACHE_SQL_MAP.containsKey(alias)) {
                    return CACHE_SQL_MAP.get(alias);
                } else {
                    source = IOTools.readText(filePath);
                    // 获取编译后的sql语句
                    source = source.replace("\r\n", "\n");
                    source = "$" + source;
                    if (mode == CACHE_MODE.NORMAL) {
                        CACHE_SQL_MAP.put(alias, source);
                    }
                    return source;
                }
            }
            case PROMPT -> {
                if (mode == CACHE_MODE.NORMAL && CACHE_PROMPT_MAP.containsKey(alias)) {
                    return CACHE_PROMPT_MAP.get(alias);
                } else {
                    source = IOTools.readText(filePath);
                    // 获取prompt内容
                    source = source.replace("\r\n", "\n");
                    if (mode == CACHE_MODE.NORMAL) {
                        CACHE_PROMPT_MAP.put(alias, source);
                    }
                    return source;
                }
            }
            default -> throw new ServiceException("错误的缓存数据类型");
        }
    }

    public enum CACHE_MODE {
        //正常模式
        NORMAL,
        //DEBUG模式，不开启缓存
        DEBUG
    }
}
