package com.af.v4.system.common.redis;

import com.af.v4.system.common.redis.utils.DistributedLockUtil;
import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.*;
import org.redisson.api.options.KeysScanOptions;
import org.redisson.codec.JsonJacksonCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

/**
 * Redis服务
 *
 * @author Mr.river
 */
@Component
public class RedisService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisService.class);

    /**
     * redisson客户端
     */
    private final RedissonClient redissonClient;
    private final DistributedLockUtil distributedLockUtil;

    public RedisService(RedissonClient REDISSON_CLIENT, DistributedLockUtil distributedLockUtil) {
        redissonClient = REDISSON_CLIENT;
        this.distributedLockUtil = distributedLockUtil;
    }

    public <T> void set(String key, T value) {
        set(key, value, -1);
    }

    public <T> void set(String key, T value, Object timeoutSec) {
        T realValue;
        if ((value instanceof JSONObject || value instanceof JSONArray) && value.toString() != null) {
            realValue = (T) value.toString();
        } else {
            realValue = value;
        }
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RBucket<T> rBucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        if (timeoutSecValue > 0) {
            rBucket.set(realValue, Duration.ofSeconds(timeoutSecValue));
        } else {
            rBucket.set(realValue);
        }
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey, value, -1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, Object timeoutSec) {
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RMap<String, T> rMap = redissonClient.getMap(hashKey);
        rMap.putAll(value);
        if (timeoutSecValue > 0) {
            rMap.expire(Duration.ofSeconds(timeoutSecValue));
        }
    }

    public JSONObject getHash(String hashKey) {
        return new JSONObject(redissonClient.getMap(hashKey));
    }

    public <T> T getHash(String hashKey, String key) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        map.put(key, value);
    }

    public void deleteHashKey(String hashKey, String key) {
        redissonClient.getMap(hashKey).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return redissonClient.getMap(hashKey).containsKey(key);
    }

    /**
     * @see DistributedLockUtil#syncLock(String, Integer, Supplier)
     */
    public <T> T syncLock(String key, Integer leaseTime, Supplier<T> supplier) {
        return distributedLockUtil.syncLock(key, leaseTime, supplier);
    }

    /**
     * @see DistributedLockUtil#syncLock(String, Supplier)
     */
    public <T> T syncLock(String key, Supplier<T> supplier) {
        return distributedLockUtil.syncLock(key, supplier);
    }

    /**
     * @see DistributedLockUtil#lock(String, Integer, Integer, Supplier)
     */
    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) throws InterruptedException {
        return distributedLockUtil.lock(key, waitTime, leaseTime, supplier);
    }

    /**
     * @see DistributedLockUtil#lock(String, Supplier)
     */
    public <T> T lock(String key, Supplier<T> supplier) throws InterruptedException {
        return distributedLockUtil.lock(key, supplier);
    }

    public void lock(String key, Integer waitTime, Integer leaseTime, Runnable runnable) throws InterruptedException {
        lock(key, waitTime, leaseTime, () -> {
            runnable.run();
            return null;
        });
    }

    public void lock(String key, Runnable runnable) throws InterruptedException {
        lock(key, () -> {
            runnable.run();
            return null;
        });
    }

    public <T> T get(String key) {
        RBucket<T> bucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        T result = bucket.get();
        T realValue;
        if (result instanceof String resultStr) {
            if (resultStr.startsWith("{") && resultStr.endsWith("}")) {
                realValue = (T) new JSONObject(resultStr);
            } else if (resultStr.startsWith("[") && resultStr.endsWith("]")) {
                realValue = (T) new JSONArray(resultStr);
            } else {
                realValue = result;
            }
        } else {
            realValue = result;
        }
        return realValue;
    }

    public boolean hasKey(String key) {
        return redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
    }

    public void delete(String key) {
        redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).delete();
    }

    public void deleteList(Iterable<String> keys) {
        keys.forEach(this::delete);
    }

    public void deleteList(JSONArray keys) {
        List<String> deleteIds = new ArrayList<>(keys.length());
        keys.forEach(item -> deleteIds.add(item.toString()));
        deleteList(deleteIds);
    }

    /**
     * 根据匹配模式获取key集合
     *
     * @param pattern 匹配模式
     * @return key集合
     */
    public Iterable<String> getKeys(String pattern) {
        RKeys keys = redissonClient.getKeys();
        return keys.getKeys(KeysScanOptions.defaults().pattern(pattern));
    }

    /**
     * 获取topic
     *
     * @param topicName topic名称
     * @return RTopic对象
     */
    public RTopic getTopic(String topicName) {
        return redissonClient.getTopic(topicName, JsonJacksonCodec.INSTANCE);
    }

    /**
     * 获取队列
     *
     * @param queueName 队列名称
     * @return RBlockingQueue 对象
     */
    public RBlockingQueue<String> getBlockingQueue(String queueName) {
        return redissonClient.getBlockingQueue(queueName);
    }
}
