package com.af.v4.system.common.redis.utils;

import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * Redis分布式锁工具
 */
@Component
public class DistributedLockUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(DistributedLockUtil.class);

    // 默认等待时间（秒）
    private static final int DEFAULT_WAIT_TIME = 5;
    // 默认锁持有时间（秒）
    private static final int DEFAULT_LEASE_TIME = 20;

    private final RedissonClient redissonClient;

    public DistributedLockUtil(RedissonClient redissonClient) {
        this.redissonClient = redissonClient;
    }

    /**
     * 获取分布式锁并执行操作（阻塞模式）。
     * <p>
     * 当前线程将阻塞直到成功获取锁，锁的最大持有时间由 leaseTime 参数决定。适用于需要确保操作
     * 必须获取锁的场景，例如长时间任务或关键业务逻辑。
     *
     * @param key       锁的唯一标识
     * @param leaseTime 锁的最大持有时间（单位：秒）
     * @param supplier  提供具体操作逻辑的函数式接口
     * @param <T>       返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T syncLock(String key, Integer leaseTime, Supplier<T> supplier) {
        try {
            return executeWithLock(key, 0, leaseTime, supplier, true);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 尝试获取分布式锁并执行操作（非阻塞模式）。
     * <p>
     * 当前线程将在 waitTime 时间内尝试获取锁。如果在此时间内未能获取锁，则抛出异常。
     * 锁的最大持有时间由 leaseTime 参数决定。适用于需要快速失败的场景。
     *
     * @param key       锁的唯一标识
     * @param waitTime  获取锁的最大等待时间（单位：秒）
     * @param leaseTime 锁的最大持有时间（单位：秒）
     * @param supplier  提供具体操作逻辑的函数式接口
     * @param <T>       返回值的类型
     * @return supplier 提供的逻辑的执行结果
     * @throws InterruptedException 如果在 waitTime 内未能获取锁
     */
    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) throws InterruptedException {
        return executeWithLock(key, waitTime, leaseTime, supplier, false);
    }

    /**
     * 默认阻塞分布式锁
     *
     * @param key      锁的唯一标识
     * @param supplier 提供具体操作逻辑的函数式接口
     * @param <T>      返回值的类型
     * @return supplier 提供的逻辑的执行结果
     */
    public <T> T syncLock(String key, Supplier<T> supplier) {
        return syncLock(key, DEFAULT_LEASE_TIME, supplier);
    }

    /**
     * 默认非阻塞分布式锁
     *
     * @param key      锁的唯一标识
     * @param supplier 提供具体操作逻辑的函数式接口
     * @param <T>      返回值的类型
     * @return supplier 提供的逻辑的执行结果
     * @throws InterruptedException 如果未能获取锁
     */
    public <T> T lock(String key, Supplier<T> supplier) throws InterruptedException {
        return lock(key, DEFAULT_WAIT_TIME, DEFAULT_LEASE_TIME, supplier);
    }

    /**
     * 执行分布式锁操作
     *
     * @param key        锁的唯一标识
     * @param waitTime   获取锁的最大等待时间（单位：秒，0 表示无限等待）
     * @param leaseTime  锁的最大持有时间（单位：秒）
     * @param supplier   提供具体操作逻辑的函数式接口
     * @param isBlocking 是否阻塞模式
     * @param <T>        返回值的类型
     * @return supplier 提供的逻辑的执行结果
     * @throws InterruptedException 如果在非阻塞模式下未能获取锁
     */
    private <T> T executeWithLock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier, boolean isBlocking) throws InterruptedException {
        RLock lock = redissonClient.getLock("LOCK-" + key);
        boolean isLock = false;

        try {
            if (isBlocking) {
                lock.lock(leaseTime, TimeUnit.SECONDS); // 阻塞模式
                isLock = true;
            } else {
                isLock = lock.tryLock(waitTime, leaseTime, TimeUnit.SECONDS); // 非阻塞模式
            }

            if (isLock) {
                return supplier.get();
            } else {
                throw new InterruptedException("获取锁超时，未能获取到锁：" + key);
            }
        } catch (InterruptedException e) {
            LOGGER.error("锁操作失败：{}，原因：{}", lock.getName(), e.getMessage());
            throw e;
        } finally {
            if (isLock && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }
}
