package com.af.v4.system.common.redis;

import com.af.v4.system.common.redis.entity.FastCacheQueueEntity;
import com.af.v4.system.common.redis.exception.LockInterruptedException;
import com.af.v4.system.common.redis.exception.LockTimeoutException;
import com.af.v4.system.common.redis.utils.DistributedLockUtil;
import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.*;
import org.redisson.api.options.KeysScanOptions;
import org.redisson.client.codec.StringCodec;
import org.redisson.codec.JsonJacksonCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.function.Supplier;

/**
 * Redis服务
 *
 * @author Mr.river
 */
@Component
public class RedisService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisService.class);

    /**
     * redisson客户端
     */
    private final RedissonClient redissonClient;
    private final DistributedLockUtil distributedLockUtil;

    public RedisService(RedissonClient REDISSON_CLIENT, DistributedLockUtil distributedLockUtil) {
        redissonClient = REDISSON_CLIENT;
        this.distributedLockUtil = distributedLockUtil;
    }

    public <T> void set(String key, T value) {
        set(key, value, -1);
    }

    public <T> void set(String key, T value, Object timeoutSec) {
        T realValue;
        if ((value instanceof JSONObject || value instanceof JSONArray) && value.toString() != null) {
            realValue = (T) value.toString();
        } else {
            realValue = value;
        }
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RBucket<T> rBucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        if (timeoutSecValue > 0) {
            rBucket.set(realValue, Duration.ofSeconds(timeoutSecValue));
        } else {
            rBucket.set(realValue);
        }
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey, value, -1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, Object timeoutSec) {
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RMap<String, T> rMap = redissonClient.getMap(hashKey);
        rMap.putAll(value);
        if (timeoutSecValue > 0) {
            rMap.expire(Duration.ofSeconds(timeoutSecValue));
        }
    }

    public JSONObject getHash(String hashKey) {
        return new JSONObject(redissonClient.getMap(hashKey));
    }

    public <T> T getHash(String hashKey, String key) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        map.put(key, value);
    }

    public void deleteHashKey(String hashKey, String key) {
        redissonClient.getMap(hashKey).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return redissonClient.getMap(hashKey).containsKey(key);
    }

    /**
     * @see DistributedLockUtil#syncLock(String, Integer, Supplier)
     */
    public <T> T syncLock(String key, Integer leaseTime, Supplier<T> supplier) throws LockInterruptedException {
        return distributedLockUtil.syncLock(key, leaseTime, supplier);
    }

    /**
     * @see DistributedLockUtil#syncLock(String, Supplier)
     */
    public <T> T syncLock(String key, Supplier<T> supplier) throws LockInterruptedException {
        return distributedLockUtil.syncLock(key, supplier);
    }

    /**
     * @see DistributedLockUtil#lock(String, Integer, Integer, Supplier)
     */
    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) throws LockTimeoutException, LockInterruptedException {
        return distributedLockUtil.lock(key, waitTime, leaseTime, supplier);
    }

    /**
     * @see DistributedLockUtil#lock(String, Supplier)
     */
    public <T> T lock(String key, Supplier<T> supplier) throws LockTimeoutException, LockInterruptedException {
        return distributedLockUtil.lock(key, supplier);
    }

    public void lock(String key, Integer waitTime, Integer leaseTime, Runnable runnable) throws LockTimeoutException, LockInterruptedException {
        lock(key, waitTime, leaseTime, () -> {
            runnable.run();
            return null;
        });
    }

    public void lock(String key, Runnable runnable) throws LockTimeoutException, LockInterruptedException {
        lock(key, () -> {
            runnable.run();
            return null;
        });
    }

    public <T> T get(String key) {
        RBucket<T> bucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        T result = bucket.get();
        T realValue;
        if (result instanceof String resultStr) {
            if (resultStr.startsWith("{") && resultStr.endsWith("}")) {
                realValue = (T) new JSONObject(resultStr);
            } else if (resultStr.startsWith("[") && resultStr.endsWith("]")) {
                realValue = (T) new JSONArray(resultStr);
            } else {
                realValue = result;
            }
        } else {
            realValue = result;
        }
        return realValue;
    }

    public <T> T getOrDefault(String key, T defaultValue) {
        T result = get(key);
        return result == null ? defaultValue : result;
    }

    public boolean hasKey(String key) {
        return redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
    }

    public void delete(String key) {
        redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).delete();
    }

    public <T> void deleteList(Iterator<T> keys) {
        RBatch batch = redissonClient.createBatch();
        while (keys.hasNext()) {
            batch.getBucket(keys.next().toString()).deleteAsync();
        }
        batch.execute();
    }

    public void deleteList(JSONArray keys) {
        deleteList(keys.iterator());
    }

    /**
     * 根据匹配模式获取key集合
     *
     * @param pattern 匹配模式
     * @return key集合
     */
    public Iterator<String> getKeys(String pattern) {
        RKeys keys = redissonClient.getKeys();
        return keys.getKeys(KeysScanOptions.defaults().pattern(pattern)).iterator();
    }

    /**
     * 使用基于 SCAN 的 Lua 脚本安全地删除大量 key
     *
     * @param pattern 匹配模式
     */
    public void deleteKeysByPatternSafely(String pattern) {
        RScript script = redissonClient.getScript(StringCodec.INSTANCE);
        String luaScript =
                """
                        local cursor = '0'\s
                        local count = 0\s
                        repeat\s
                            local result = redis.call('SCAN', cursor, 'MATCH', ARGV[1], 'COUNT', ARGV[2])\s
                            cursor = result[1]\s
                            local keys = result[2]\s
                            if #keys > 0 then\s
                                redis.call('UNLINK', unpack(keys))\s
                                count = count + #keys\s
                            end\s
                        until cursor == '0'\s
                        return count""";

        // ARGV[1] 是 pattern, ARGV[2] 是 COUNT
        script.eval(RScript.Mode.READ_WRITE, luaScript, RScript.ReturnType.INTEGER,
                Collections.emptyList(), pattern, String.valueOf(1000));
    }

    /**
     * 获取topic
     *
     * @param topicName topic名称
     * @return RTopic对象
     */
    public RTopic getTopic(String topicName) {
        return redissonClient.getTopic(topicName, JsonJacksonCodec.INSTANCE);
    }

    /**
     * 获取队列
     *
     * @param queueName 队列名称
     * @return RBlockingQueue 对象
     */
    public RBlockingQueue<String> getBlockingQueue(String queueName) {
        return redissonClient.getBlockingQueue(queueName);
    }

    /**
     * 推送数据到 Redis 队列尾部
     *
     * @param queueName 队列名称
     * @param data      队列数据
     */
    public <T> void pushToQueue(String queueName, T data) {
        RBlockingQueue<T> queue = redissonClient.getBlockingQueue(queueName);
        try {
            queue.put(data);
            LOGGER.info(">>> 数据已推送至队列：{}, 内容：{}", queueName, data);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.error("推送数据中断：{}", e.getMessage(), e);
        }
    }

    /**
     * 推送数据到 Redis 队列尾部
     *
     * @param queueName 队列名称
     * @param data      队列数据
     */
    public void pushToQueue(String queueName, JSONObject data) {
        FastCacheQueueEntity fastCacheQueue = new FastCacheQueueEntity.Builder()
                .maxRetryTimes(data.optInt("maxRetryTimes", 3))
                .data(data.optJSONObject("data")).build();
        pushToQueue(queueName, fastCacheQueue);
    }

    /**
     * 阻塞式获取队列头部数据（队列为空时自动等待）
     *
     * @param queueName 队列名称
     * @return 取出的数据
     */
    public <T> T blockingPopFromQueue(String queueName) {
        RBlockingQueue<T> queue = redissonClient.getBlockingQueue(queueName);
        try {
            return queue.take();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.error(">>> 获取队列数据中断：{}", e.getMessage(), e);
            return null;
        }
    }

    /**
     * 原子递增（或递减）指定 key 的值。
     * 1. 若 key 不存在，则先初始化为 {@code initialValue}，再执行递增/递减；
     * 2. 自动设置过期时间（仅第一次写入时生效），避免遗留脏 key；
     * 3. 线程安全、分布式安全。
     *
     * @param key           缓存 key
     * @param delta         步长，正数为递增，负数为递减
     * @param initialValue  首次初始化时的值（通常写 0 或 1）
     * @param expireSeconds 过期时间（秒）；小于等于 0 表示永不过期
     * @return 递增/递减后的最新值
     */
    public long incr(String key, long delta, long initialValue, long expireSeconds) {
        RAtomicLong atomic = redissonClient.getAtomicLong(key);
        // 用是否存在判断是不是第一次，只有第一次才设置过期时间，避免每次覆盖
        if (!atomic.isExists()) {
            atomic.set(initialValue);
            if (expireSeconds > 0) {
                atomic.expire(Duration.ofSeconds(expireSeconds));
            }
        }
        return atomic.addAndGet(delta);
    }
}
