package com.af.v4.system.common.redis.core;

import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.redis.annotation.BeforeRetry;
import com.af.v4.system.common.redis.annotation.FastCacheQueue;
import com.af.v4.system.common.redis.annotation.FilterCondition;
import com.af.v4.system.common.redis.annotation.MaxRetry;
import com.af.v4.system.common.redis.entity.FastCacheQueueEntity;
import jakarta.annotation.PreDestroy;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @Author: 小六
 * @Description: Redis 队列消费者统一管理器
 */
@Component
public class FastCacheQueueManager {

    private static final Logger log = LoggerFactory.getLogger(FastCacheQueueManager.class);

    private final RedisService redisService;
    private final ApplicationContext context;

    private final Map<String, Thread> consumerThreads = new ConcurrentHashMap<>();
    private final Map<String, Boolean> runningFlags = new ConcurrentHashMap<>();

    public FastCacheQueueManager(RedisService redisService, ApplicationContext context) {
        this.redisService = redisService;
        this.context = context;
    }

    /**
     * 遍历所有 FastCacheQueueProcess Bean，绑定注解信息，注册队列消费者
     */
    public void registerFromConfig(JSONObject config, FastCacheQueueManager manager) {
        JSONArray queueArr = config.optJSONArray("queue");
        if (queueArr == null || queueArr.isEmpty()) return;

        Map<String, FastCacheQueueProcess> processMap = new HashMap<>();

        for (FastCacheQueueProcess bean : context.getBeansOfType(FastCacheQueueProcess.class).values()) {
            FastCacheQueue anno = bean.getClass().getAnnotation(FastCacheQueue.class);
            if (anno == null) continue;
            for (String queue : anno.value()) {
                processMap.put(queue, bean);
            }
        }

        for (int i = 0; i < queueArr.length(); i++) {
            JSONObject item = queueArr.getJSONObject(i);
            String queueName = item.optString("name");
            int consumers = item.optInt("consumers", 1);

            FastCacheQueueProcess processor = processMap.get(queueName);
            if (processor == null) {
                log.warn(">>> 未找到处理器: {}，跳过队列注册", queueName);
                continue;
            }

            ConsumerRegistration reg = new ConsumerRegistration(queueName)
                    .onProcess(processor::onProcess)
                    .onSuccess(processor::onSuccess)
                    .onFailure(processor::onFailure);

            // 自动读取 @FilterCondition、@BeforeRetry、@MaxRetry
            for (Method method : processor.getClass().getDeclaredMethods()) {
                if (method.isAnnotationPresent(FilterCondition.class)) {
                    method.setAccessible(true);
                    reg.filter(data -> {
                        try {
                            return (boolean) method.invoke(processor, data);
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    });
                }
                if (method.isAnnotationPresent(BeforeRetry.class)) {
                    method.setAccessible(true);
                    reg.beforeRetry((data, retryContext) -> {
                        try {
                            method.invoke(processor, data, retryContext);
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    });
                }
            }

            // 类上的 @MaxRetry
            MaxRetry clsRetryAnn = processor.getClass().getAnnotation(MaxRetry.class);
            if (clsRetryAnn != null) reg.maxRetries(clsRetryAnn.value());

            reg.registerWith(manager, consumers);
        }
    }

    /**
     * 注册消费者线程（内部调用）
     */
    protected void register(String queueName, int threadNumber, ConsumerRegistration reg) {
        String threadKey = STR."\{queueName}$\{threadNumber}";
        runningFlags.put(threadKey, true);

        Runnable task = () -> {
            log.info(">>> 启动消费线程 ▶️ {}", threadKey);
            while (Boolean.TRUE.equals(runningFlags.get(threadKey))) {
                try {
                    Object raw = redisService.blockingPopFromQueue(queueName);
                    if (reg.verbose) log.info(">>> 队列 {} 返回数据：{}", queueName, raw);
                    if (!(raw instanceof FastCacheQueueEntity entity)) {
                        if (reg.verbose)
                            log.warn(">>> 队列 {} 返回数据不是 FastCacheQueueEntity，跳过处理", queueName);
                        continue;
                    }

                    JSONObject data = entity.getData();
                    int retry = entity.getCurrRetryTimes();
                    int maxRetry = entity.getMaxRetryTimes() > 0 ? entity.getMaxRetryTimes() : reg.defaultMaxRetries;

                    if (reg.filter != null && !reg.filter.test(data)) {
                        if (reg.verbose) log.info(">>> 消息被拦截 ▶️ {}", threadKey);
                        continue;
                    }

                    try {
                        if (reg.verbose) log.info("消费开始 ▶️ {}", threadKey);
                        Object result = null;
                        if (reg.onProcess != null) {
                            if (reg.verbose) log.info(">>> 消费开始 ▶️ {}", threadKey);
                            result = reg.onProcess.apply(data);
                        }
                        reg.onSuccess.accept(data, result);
                        if (reg.verbose) log.info(">>> 消费成功 ▶️ {}", threadKey);
                    } catch (Exception e) {
                        retry++;
                        boolean allowRetry = maxRetry > 0 && retry < maxRetry && (reg.condition == null || reg.condition.test(data, e));

                        if (allowRetry) {
                            FastCacheQueueEntity retryEntity = new FastCacheQueueEntity.Builder()
                                    .data(data)
                                    .currRetryTimes(retry)
                                    .maxRetryTimes(maxRetry)
                                    .build();

                            if (reg.beforeRetry != null) {
                                JSONObject retryContext = new JSONObject()
                                        .put("retryCount", retry)
                                        .put("maxRetry", maxRetry)
                                        .put("exception", e.getMessage());
                                reg.beforeRetry.accept(data, retryContext);
                            }

                            redisService.pushToQueue(queueName, retryEntity);
                        } else {
                            log.error("🛑 >>> 消费失败 ▶️ {}，异常：{}", threadKey, e.getMessage());
                            if (reg.onFailure != null) reg.onFailure.accept(data, e);
                            if (reg.requeueOnFailure) redisService.pushToQueue(queueName, entity);
                        }
                    }
                } catch (Exception ex) {
                    log.error(">>> 线程异常 ▶️ {}，原因：{}", threadKey, ex.getMessage());
                }
            }
            log.info(">>> 消费线程退出 ▶️ {}", threadKey);
        };

        Thread thread = new Thread(task, threadKey);
        thread.start();
        consumerThreads.put(threadKey, thread);
    }

    /**
     * 关闭指定队列
     */
    public void shutdownQueue(String queueName) {
        log.info(">>> 停止队列 ▶️ {}", queueName);
        consumerThreads.keySet().stream()
                .filter(k -> k.startsWith(STR."\{queueName}$"))
                .forEach(k -> {
                    runningFlags.put(k, false);
                    Thread t = consumerThreads.remove(k);
                    if (t != null) {
                        try {
                            t.join(1000);
                            log.info(">>> 已关闭线程 ▶️ {}", k);
                        } catch (InterruptedException ignored) {
                        }
                    }
                });
    }

    /**
     * 应用退出时关闭所有线程
     */
    @PreDestroy
    public void shutdownAll() {
        log.info(">>> 正在关闭所有队列线程...");
        runningFlags.keySet().forEach(k -> runningFlags.put(k, false));
        consumerThreads.values().forEach(t -> {
            try {
                t.join(1000);
            } catch (InterruptedException ignored) {
            }
        });
        log.info(">>> 所有线程已关闭");
    }
}
