package com.af.v4.system.common.redis;

import org.json.JSONArray;
import org.json.JSONObject;
import org.redisson.api.*;
import org.redisson.codec.JsonJacksonCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * Redis服务
 *
 * @author Mr.river
 */
@Component
public class RedisService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RedisService.class);

    /**
     * redisson客户端
     */
    private final RedissonClient redissonClient;

    public RedisService(RedissonClient REDISSON_CLIENT) {
        redissonClient = REDISSON_CLIENT;
    }

    public <T> void set(String key, T value) {
        set(key, value, -1);
    }

    public <T> void set(String key, T value, Object timeoutSec) {
        T realValue;
        if (value instanceof JSONObject || value instanceof JSONArray) {
            realValue = (T) value.toString();
        } else {
            realValue = value;
        }
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RBucket<T> rBucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        if (timeoutSecValue > 0) {
            rBucket.set(realValue, Duration.ofSeconds(timeoutSecValue));
        } else {
            rBucket.set(realValue);
        }
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey, value, -1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, Object timeoutSec) {
        long timeoutSecValue = Long.parseLong(timeoutSec.toString());
        RMap<String, T> rMap = redissonClient.getMap(hashKey);
        rMap.putAll(value);
        if (timeoutSecValue > 0) {
            rMap.expire(Duration.ofSeconds(timeoutSecValue));
        }
    }

    public JSONObject getHash(String hashKey) {
        return new JSONObject(redissonClient.getMap(hashKey));
    }

    public <T> T getHash(String hashKey, String key) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        RMap<String, T> map = redissonClient.getMap(hashKey);
        map.put(key, value);
    }

    public void deleteHashKey(String hashKey, String key) {
        redissonClient.getMap(hashKey).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return redissonClient.getMap(hashKey).containsKey(key);
    }

    public <T> T syncLock(String key, Supplier<T> supplier) {
        return syncLock(key, 30, supplier);
    }

    public <T> T syncLock(String key, Integer leaseTime, Supplier<T> supplier) {
        RLock lock = redissonClient.getLock("LOCK-" + key);
        try {
            lock.lock(leaseTime, TimeUnit.SECONDS);
            return supplier.get();
        } finally {
            if (lock != null && lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }

    public void lock(String key, Runnable runnable) throws InterruptedException {
        lock(key, 5, 20, runnable);
    }

    public void lock(String key, Integer waitTime, Integer leaseTime, Runnable runnable) throws InterruptedException {
        lock(key, waitTime, leaseTime, () -> {
            runnable.run();
            return null;
        });
    }

    public <T> T lock(String key, Supplier<T> supplier) throws InterruptedException {
        return lock(key, 5, 20, supplier);
    }

    public <T> T lock(String key, Integer waitTime, Integer leaseTime, Supplier<T> supplier) throws InterruptedException {
        RLock lock = redissonClient.getLock("LOCK-" + key);
        boolean isLock;
        try {
            isLock = lock.tryLock(waitTime, leaseTime, TimeUnit.SECONDS);
            if (isLock) {
                return supplier.get();
            } else {
                throw new InterruptedException("操作失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            LOGGER.error("分布式锁：{}获取失败", lock.getName());
            throw e;
        } finally {
            if (lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }


    public <T> T get(String key) {
        RBucket<T> bucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
        T result = bucket.get();
        T realValue;
        if (result instanceof String resultStr) {
            if (resultStr.startsWith("{") && resultStr.endsWith("}")) {
                realValue = (T) new JSONObject(resultStr);
            } else if (resultStr.startsWith("[") && resultStr.endsWith("]")) {
                realValue = (T) new JSONArray(resultStr);
            } else {
                realValue = result;
            }
        } else {
            realValue = result;
        }
        return realValue;
    }

    public boolean hasKey(String key) {
        return redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
    }

    public void delete(String key) {
        redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).delete();
    }

    public void deleteList(Iterable<String> keys) {
        keys.forEach(this::delete);
    }

    public void deleteList(JSONArray keys) {
        List<String> deleteIds = new ArrayList<>(keys.length());
        keys.forEach(item -> deleteIds.add(item.toString()));
        deleteList(deleteIds);
    }

    /**
     * 根据匹配模式获取key集合
     *
     * @param Pattern 匹配模式
     * @return key集合
     */
    public Iterable<String> getKeys(String Pattern) {
        RKeys keys = redissonClient.getKeys();
        return keys.getKeysByPattern(Pattern);
    }

    /**
     * 获取topic
     *
     * @param topicName topic名称
     * @return RTopic对象
     */
    public RTopic getTopic(String topicName) {
        return redissonClient.getTopic(topicName, JsonJacksonCodec.INSTANCE);
    }
}
